;;;
;;; Simple Parallel Coordinates Plot
;;;

(defproto parallel-plot-proto '(v) () graph-proto)

(send parallel-plot-proto :title "Parallel Plot")

(defmeth parallel-plot-proto :isnew (m &rest args)
  (setf (slot-value 'v) 0)
  (apply #'call-next-method (+ 1 m) args)
  (send self :content-variables m 0))

(defmeth parallel-plot-proto :current-axis (&optional (i nil set)
                                            &key (draw t))
  (when set
        (setf (slot-value 'v) i)
        (let* ((n (send self :num-points))
               (m (- (send self :num-variables) 1))
               (i (max 0 (min i (- m 1)))))
          (when (< 0 n) (send self :point-coordinate m (iseq n) i))
          (send self :content-variables m i))
        (if draw (send self :redraw)))
  (slot-value 'v))
        
(defmeth parallel-plot-proto :choose-current-axis ()
  (let* ((choices (mapcar #'(lambda (x) (format nil "~d" x))
                          (iseq (- (send self :num-variables) 1))))
         (v (choose-item-dialog "Current Axis" choices :initial
                                (send self :current-axis))))
    (if v (send self :current-axis v))))
    
(defmeth parallel-plot-proto :menu-template ()
  (flet ((action () (send self :choose-current-axis)))
    (let ((item (send menu-item-proto :new "Current Variable ..."
                      :action #'action)))
      (append (call-next-method) (list item)))))

(defmeth parallel-plot-proto :adjust-to-data (&key (draw t))
  (call-next-method :draw nil)
  (let ((m (- (send self :num-variables) 1)))
    (when (null (send self :scale-type))
          (flet ((expand-range (i)
                   (let* ((range (send self :range i))
                          (mid (mean range))
                          (half (- (second range) (first range)))
                          (low (- mid (* 0.55 half)))
                          (high (+ mid (* 0.55 half))))
                     (send self :range i low high :draw nil))))
            (dotimes (i m) (expand-range i))))
    (send self :scale m 1 :draw nil)
    (send self :center m 0 :draw nil)
    (send self :range m -.1 (- m .9) :draw nil)))
        
(defmeth parallel-plot-proto :add-points (data &rest args &key (draw t))
  (let ((n (length (first data))))
    (apply #'call-next-method (append data (list (repeat 0 n)))
           :draw nil args)
    (send self :current-axis (send self :current-axis) :draw draw)))

(defmeth parallel-plot-proto :add-lines (&rest args)
  (error "Lines are not meaningful for this plot"))

(defmeth parallel-plot-proto :resize ()
  (call-next-method)
  (let ((height (fourth (send self :content-rect)))
        (m (- (send self :num-variables) 1)))
    (send self :canvas-range (iseq m) 0 height)))

(defmeth parallel-plot-proto :draw-parallel-point (i)
  (let* ((points (if (numberp i) (list i) i))
         (width (third (send self :content-rect)))
         (origin (send self :content-origin))
         (x-origin (first origin))
         (y-origin (second origin))
         (m (- (send self :num-variables) 1))
         (gap (/ width (+ (- m 1) .2)))
         (indices (iseq m))
         (xvals (+ x-origin (round (* gap (+ .1 indices)))))
         (oldcolor (send self :draw-color)))
    (dolist (i points)
      (when (send self :point-showing i)
            (let* ((color (send self :point-color i))
                   (yvals (- y-origin
                             (send self :point-canvas-coordinate indices i)))
                   (poly (transpose (list xvals yvals))))
              (if color (send self :draw-color color))
              (send self :frame-poly poly)
              (if color (send self :draw-color oldcolor)))))))

(defmeth parallel-plot-proto :redraw-content ()
  (let ((indices (iseq (send self :num-points))))
    (send self :start-buffering)
    (call-next-method)
    (send self :draw-parallel-point indices)
    (send self :buffer-to-screen)))

(defun parallel-plot (data &rest args &key point-labels)
  (let ((graph (apply #'send parallel-plot-proto
                      :new (length data) :draw nil args)))
    (if point-labels
        (send graph :add-points data :point-labels point-labels :draw nil)
        (send graph :add-points data :draw nil))
    (send graph :adjust-to-data :draw nil)
    graph))


;;;
;;; Parallel Coordinates Grand Tour
;;;

(defproto parallel-tour-proto '(angle) ()
  (list tour-mixin parallel-plot-proto))

(defmeth parallel-tour-proto :angle (&optional (val nil set))
  (when set (setf (slot-value 'angle) val))
  (slot-value 'angle))

(send parallel-tour-proto :angle .1)

(defmeth parallel-tour-proto :num-tour-variables ()
  (- (send self :num-variables) 1))

(send parallel-tour-proto :slot-value 'scale-type 'variable)

(defun tour-parallel-plot (data &rest args &key point-labels)
  (let ((graph (apply #'send parallel-tour-proto :new (length data) args)))
    (if point-labels
        (send graph :add-points data :point-labels point-labels :draw nil)
        (send graph :add-points data :draw nil))
    (send graph :adjust-to-data :draw nil)
    graph))
