(load-example "plotcontrols")

(defun bc (x p)
  (let* ((bcx (if (< (abs p) .0001)
                  (log x)
                  (/ (^ x p) p)))
         (max (max bcx))
         (min (min bcx)))
    (/ (- bcx min) (- max min))))

(defun bcr (x p)
  (if (< (abs p) 0.0001) (log x) (/ (^ x p) p)))

(defun sc (x)
  (let ((min (min x))
        (max (max x)))
    (/ (- x min) (- max min))))

(let* ((names '("Length" "Width" "Height" "Shell" "Mass"))
       (data (transpose (split-list '(
    318  68 158 345  47
    312  56 148 290  52
    265  46 124 167  27
    222  38 104  67  13
    274  51 143 238  31
    216  35  99  68  14
    217  34 109  75  15
    202  32  96  54   4
    272  44 119 128  23
    273  49 123 150  32
    260  48 135 117  30
    276  47 133 190  26
    270  50 126 160  24
    280  52 130 212  31
    262  50 134 208  31
    312  61 120 235  42
    220  34  94  52   9
    212  32 102  74  13
    196  28  85  42   7
    226  38 104  69  13
    284  61 134 268  50
    320  60 137 323  39
    331  60 140 359  47
    276  46 126 167  40
    186  30  92  33   5
    213  35  98  51  12
    291  47 130 170  26
    298  54 137 224  32
    287  55 140 238  40
    230  40 106  68  16
    293  57 135 208  33
    298  48 135 167  28
    290  47 134 187  28
    282  52 135 191  42
    221  37 104  58  15
    287  54 135 180  27
    228  46 129 188  33
    210  33 107  65  14
    308  58 131 299  29
    265  48 124 159  26
    270  44 124 145  25
    208  33  99  54   9
    277  45 123 129  18
    241  39 110 104  23
    219  38 105  66  13
    170  27  87  24   6
    150  21  75  19   6
    132  20  65  10   1
    175  30  86  36   8
    150  22  69  18   5
    162  25  79  20   6
    252  47 124 133  22
    275  48 131 179  24
    224  36 107  69  13
    211  33 100  59  11
    254  46 126 120  18
    234  37 114  72  17
    221  37 108  74  15
    167  27  80  27   7
    220  36 106  52  14
    227  35 118  76  14
    177  25  83  25   8
    230  47 112 125  18
    288  46 132 138  24
    275  54 127 191  29
    273  42 120 148  21
    246  37 110  90  17
    250  43 115 120  17
    290  48 131 203  34
    226  35 111  64  16
    269  45 121 124  22
    267  48 121 153  24
    263  48 123 151  19
    217  36 104  68  13
    188  33  93  51  10
    152  25  76  19   5
    227  38 112  88  15
    216  25 110  53  12
    242  45 112  61  12
    260  44 123 133  24
    196  35 101  68  15
    220  36 105  64  16
) 5))))

(mapcar #'(lambda (x y) (set (intern (string-upcase x)) y)) names data))

(setf mussels (regression-model (list height shell) mass :print nil))

(defmeth mussels :simple-bc ()
  (let* ((n (send self :num-cases))
	 (nqr (sc (normal-quant (/ (iseq 1 n) (+ 1 n)))))
         (idx (iseq n))
	 (r (send self :residuals))
	 (p (plot-points (select nqr (rank r)) (sc r))))
    (defmeth p :change-power (pow)
      (send mussels :y (bcr mass pow))
      (let* ((r (send mussels :residuals))
             (r-nqr (select nqr (rank r))))
        (send self :point-coordinate 0 idx r-nqr)
        (send self :point-coordinate 1 idx (sc r))
        (send self :redraw-content)
	(pause 2)))
    (let ((s (interval-slider-dialog 
              '(-1 2)  :action #'(lambda (pow) (send p :change-power pow)))))
      (send s :value 1)
      (send p :add-subordinate s))))

(defproto scatterplot-power-control-proto
  '(variable data model) () slider-control-proto)

(defmeth scatterplot-power-control-proto :isnew (var dat &optional
						     m
						     (points 31))
  (let* ((seq (rseq -1 2 points))
	 (aseq1 (abs (- seq 1)))
	 (one (position (min aseq1) aseq1)))
    (setf (slot-value 'variable) var)
    (setf (slot-value 'data) dat)
    (setf (slot-value 'model) m)
    (call-next-method seq :location '(10 10) :title "Power")
    (setf (slot-value 'index) one)))

(defmeth scatterplot-power-control-proto :resize ()
  (let* ((graph (send self :graph))
	 (height (fourth (send graph :content-rect)))
	 (n (send graph :num-variables))
	 (var (slot-value 'variable)))
    (setf (slot-value 'location)
	  (list 10
		(round (- (* (- 1 (/ (+ 0.5 var) n)) height)
			  (* 0.5 (second (send self :size)))))))))

(defmeth scatterplot-power-control-proto :draw-indicator (&optional index)
  (let* ((graph (send self :graph))
         (loc (send self :location))
         (loc-x (first loc))
         (loc-y (second loc))
         (w (first (send self :size)))
         (min (send self :min))
         (max (send self :max))
         (index (if index index (send self :index)))
         (val (floor (* (- w 7) (/ (- index min) (- max min))))))
    (when graph
          (let ((tw (send graph :text-width (send self :title))))
            (send graph :start-buffering)
            (send graph :erase-rect (+ 1 tw loc-x) loc-y (- w tw) 20)
            (send graph :draw-text 
                  (format nil "~,2f" (elt (send self :display) index))
                  (+ loc-x w) (+ loc-y 15) 2 0)
            (send graph :buffer-to-screen (+ 1 tw loc-x) loc-y (- w tw) 20))
          (send graph :erase-rect (+ 1 loc-x) (+ 21 loc-y) (- w 2) 8)
          (send graph :paint-rect (+ 1 loc-x val) (+ 21 loc-y) 5 8))))

(defmeth scatterplot-power-control-proto :do-action (v)
  (let ((graph (send self :graph)))
    (when graph
	  (let* ((i (iseq (send graph :num-points)))
		 (var (slot-value 'variable))
		 (dat (slot-value 'data))
		 (m (slot-value 'model))
		 (new-dat (bc dat v)))
	    (send graph :point-coordinate var i new-dat)
	    (when m
		  (let ((n (send graph :num-variables)))
		    (if (= var (- n 1))
			(send m :y new-dat)
		        (let ((x (copy-array (send m :x))))
			  (setf (select x i var) (bind-columns new-dat))
			  (send m :x x))))
		  (send m :compute))
	    (send graph :redraw)
	    (pause 2)))))

(defproto r-squared-overlay-proto '(model) '() graph-overlay-proto)

(defmeth r-squared-overlay-proto :isnew (m &rest args)
  (setf (slot-value 'model) m)
  (apply #'call-next-method args))

(defmeth r-squared-overlay-proto :redraw ()
  (let* ((graph (send self :graph))
         (crect (send graph :content-rect))
         (left (+ 10 (first (send graph :margin)) (third crect)))
         (top (+ 10 (second crect)))
         (width 10)
         (height (- (fourth crect) 20))
         (value (send (slot-value 'model) :r-squared))
         (vtop (+ top (round (* height (max 0 (min 1 (- 1 value)))))))
         (vheight (- (+ top height) vtop)))
    (send graph :start-buffering)
    (send graph :erase-rect (+ left 1) (+ top 1) (- width 2) (- height 2))
    (send graph :frame-rect left top width height)
    (send graph :paint-rect left vtop width vheight)
    (send graph :buffer-to-screen left top width height)))

(defmeth mussels :scatmat-bc ()
  (let ((p (scatterplot-matrix (mapcar #'sc (list height shell mass))
			       :variable-labels '("Height" "Shell" "Mass")
			       :show nil)))
    (send p :margin 120 0 30 0)
    (apply #'send p :size (+ (send p :size) '(150 0)))
    (send p :resize)
    (send p :add-control
	  (send scatterplot-power-control-proto :new 2 mass self))
    (send p :add-control
	  (send scatterplot-power-control-proto :new 1 shell self))
    (send p :add-control
	  (send scatterplot-power-control-proto :new 0 height self))
    (send p :add-overlay (send r-squared-overlay-proto :new self))
    (defmeth p :redraw ()
      (send self :start-buffering)
      (call-next-method)
      (send self :redraw-overlays)
      (send self :buffer-to-screen))
    (send p :show-window)))

(send mussels :add-slot 'normal-residual-plot)

(let* ((n (send mussels :num-cases))
       (nqr (sc (normal-quant (/ (iseq 1 n) (+ 1 n)))))
       (idx (iseq n)))
  (defmeth mussels :normal-residual-plot ()
    (unless (member (slot-value 'normal-residual-plot) (active-graph-windows))
	    (let* ((r (send self :residuals))
		   (p (plot-points (select nqr (rank r)) (sc r) :show nil)))
	      (send p :x-axis nil)
	      (send p :y-axis nil)
	      (send p :scale-type 'fixed)
	      (send p :show-window)
	      (setf (slot-value 'normal-residual-plot) p))))
		    
  (defmeth mussels :compute ()
    (call-next-method)
    (let ((p (slot-value 'normal-residual-plot)))
      (when (member p (active-graph-windows))
	    (let* ((r (send self :residuals))
		   (r-nqr (select nqr (rank r))))
	      (send p :point-coordinate 0 idx r-nqr)
	      (send p :point-coordinate 1 idx (sc r))
	      (send p :redraw-content)
	      (pause 2))))))

(let ((mussels-menu (send menu-proto :new "Mussels"))
      (description "
Horse mussels, { it Atrinia}, were sampled from the Marlborough
Sounds.  The response is the mussels' Muscle Mass (Mass) in grams, the
edible portion of the mussel.  The possible predictors are all
characteristics of the mussels' shell:

  Length = shell length in mm
  Width  = shell width in mm
  Height = shell height in mm
  Shell  = shell mass in g
"))
  (send mussels-menu :append-items
	(send menu-item-proto :new "Describe"
	      :action #'(lambda () (format t "~a~%" description)))
	(send dash-item-proto :new)
	(send menu-item-proto :new "Simple BC"
	      :action #'(lambda () (send mussels :simple-bc)))
	(send menu-item-proto :new "Scatmat BC"
	      :action #'(lambda () (send mussels :scatmat-bc)))
	(send menu-item-proto :new "Normal-Residual QQ"
	      :action #'(lambda () (send mussels :normal-residual-plot)))
	(send dash-item-proto :new)
	(send menu-item-proto :new "Remove"
	      :action #'(lambda () (send mussels-menu :remove))))
  (send mussels-menu :install))
