registerMethod <- function(genname, class, method) {
    genfun <- get(genname, envir = parent.frame())
    if (typeof(genfun) == "closure")
        defenv <- environment(genfun)
    else defenv <- .BaseNamespaceEnv
    if (! exists(".MethodsTable", envir = defenv, inherits = FALSE))
        assign(".MethodsTable", new.env(hash = TRUE, parent = NULL),
               envir = defenv)
    table <- get(".MethodsTable", envir = defenv, inherits = FALSE)
    assign(paste(genname, class, sep = "."), method, envir = table)
}
getNamespace <- function(name) {
    ns <- .Internal(getRegisteredNamespace(as.name(name)))
    if (! is.null(ns)) ns
    else loadNamespace(name)
}
registeredNamespaces <- function()
    ls(env = .Internal(getNamespaceRegistry()))
asNamespace <- function(ns) {
    if (is.character(ns)) getNamespace(ns)
    else if (is.environment(ns) &&
             (identical(ns, .BaseNamespaceEnv) ||
              exists(".__NAMESPACE__.", env = ns, inherits = FALSE)))
        ns
    else browser() # stop("bad namespace argument")
}
makeImportExportNames <- function(spec) {
    old <- as.character(spec)
    new <- names(spec)
    if (is.null(new)) new <- old
    else new[new==""] <- old[new==""]
    names(old) <- new
    old
}
attachNamespace <- function(ns, prefix="namespace") {
    ns <- asNamespace(ns)
    exports <- namespaceExports(ns)
    nsget <- function(n) namespaceGet(ns, n)
    frame <- structure(lapply(exports, nsget), names=exports)
    framename <- paste(prefix, namespaceName(ns),sep=":")
    #**** FIXME: make read-only??
    attach(frame, name = framename)
}
namespaceFilePath <- function(package, package.lib)
    file.path(package.lib, package, "NAMESPACE")
packageHasNamespace <- function(package, package.lib)
    file.exists(namespaceFilePath(package, package.lib))
parseNamespaceFile <- function(package, package.lib) {
    nsFile <- namespaceFilePath(package, package.lib)
    if (! file.exists(nsFile)) {
        fQuote <- function(s) paste("`", s, "'", sep = "")
        stop(paste("package", fQuote(package), "has no NAMESPACE file"))
    }
    exports <- character(0)
    export.patterns <- character(0)
    imports <- list()
    for (e in parse(nsFile))
        switch(as.character(e[[1]]),
               export = {
                   exp <- e[-1]
                   exp <- structure(as.character(exp), names=names(exp))
                   exports <- c(exports, exp)
               },
               export.pattern = {
                   pat <- as.character(e[-1])
                   export.patterns <- c(pat, export.patterns)
               },
               import = imports <- c(imports,as.list(as.character(e[-1]))),
               import.from = {
                   imp <- e[-1]
                   ivars <- imp[-1]
                   inames <- names(ivars)
                   imp <- list(as.character(imp[1]),
                               structure(as.character(ivars), names=inames))
                   imports <- c(imports, list(imp))
               },
               stop(paste("unknown namespace directive:", deparse(e))))
    list(imports=imports, exports=exports, export.patterns = export.patterns)
}
loadNamespace <- function (package, lib.loc = .lib.loc,
                            keep.source = getOption("keep.source.pkgs")) {
    package <- as.character(package)
    ns <- .Internal(getRegisteredNamespace(as.name(package)))
    if (! is.null(ns))
        ns
    else {
        fQuote <- function(s) paste("`", s, "'", sep = "")
        run.First.lib <- function(fun, env, package, package.lib) {
            tt <- try(fun(package.lib, package))
            if (inherits(tt, "try-error"))
                stop(".First.lib failed")
        }
        pkgpath <- .find.package(package, lib.loc, quiet = TRUE)
        if (length(pkgpath) == 0)
            stop(paste("There is no package called", fQuote(package)))
        package.lib <- dirname(pkgpath)

        #**** FIXME: make sure package has a namespace file
        # create namespace and process imports
        nsInfo <- parseNamespaceFile(package, package.lib)
        ns <- makeNamespace(package)
        #**** FIXME: arrange to unregister if there are errors?
        #**** Need cycle detection
        for (i in nsInfo$imports) {
            if (is.character(i))
                namespaceImport(ns, loadNamespace(i, lib.loc, keep.source))
            else
                namespaceImportFrom(ns, loadNamespace(i[[1]], lib.loc,
                                                      keep.source),
                                    i[[2]])
        }
        env <- namespaceEnv(ns)

        codeFile <- file.path(package.lib, package, "R", package)
        if (file.exists(codeFile)) 
            sys.source(codeFile, env, keep.source = keep.source)
        else warning(paste("Package ", fQuote(package), "contains no R code"))

        if (exists(".First.lib", envir = env, inherits = FALSE)) {
             firstlib <- get(".First.lib", envir = env, inherits = FALSE)
             run.First.lib(firstlib, env, package, package.lib)
         }
         firstlib <- getOption(".First.lib")[[package]]
         if (!is.null(firstlib))
             run.First.lib(firstlib, env, package, package.lib)

         # export, seal, and register
         exports <- nsInfo$exports
         for (p in nsInfo$export.patterns)
             exports <- c(ls(env, pat = p, all = TRUE), exports)
         namespaceExport(ns, exports)
         sealNamespace(ns)

         ns
     }                               
}
topenv <- function(envir = parent.frame()) {
    while (! is.null(envir)) {
        if (identical(envir, .GlobalEnv))
            return(.GlobalEnv)
        else if (exists(".__NAMESPACE__.", envir = envir, inherits = FALSE))
            return(envir)
        envir <- parent.env(envir)
    }
    return(.GlobalEnv)
}
makeNamespace <- function(name) {
    impenv <- new.env(parent = .BaseNamespaceEnv, hash = TRUE)
    env <- new.env(parent = impenv, hash = TRUE)
    name <- as.character(as.name(name))
    assign(".__NAMESPACE__.", name, env = env)
    assign(".__EXPORTS__.", character(0), env = env)
    assign(".__IMPORTS__.", list("base"=TRUE), env = env)
    .Internal(registerNamespace(name, env))
    env
}
namespaceImport <- function(self, ...) {
    if (namespaceIsSealed(self)) stop("namespace is sealed")
    for (ns in list(...)) {
        ns <- asNamespace(ns)
        namespaceImportFrom(self, ns)
    }
}
namespaceImportFrom <- function(self, ns, vars) {
    #**** FIXME: need to worry about uniqueness of names?
    if (namespaceIsSealed(self))
        stop("cannot import into a sealed namespace")
    if (identical(namespaceEnv(self), .BaseNamespaceEnv))
        stop("cannot import into base name space")
    ns <- asNamespace(ns)
    if (missing(vars)) impvars <- namespaceExports(ns)
    else impvars <- vars
    impvars <- makeImportExportNames(impvars)
    impnames <- names(impvars)
    impenv <- parent.env(namespaceEnv(self))
    for (i in seq(along = impvars))
        assign(impnames[i], namespaceGet(ns, impvars[i]), env=impenv)
    if (missing(vars)) imp <- TRUE
    else imp <- impvars
    imp <- structure(list(imp), names = namespaceName(ns))
    imports <- get(".__IMPORTS__.", env = namespaceEnv(self))
    assign(".__IMPORTS__.", c(imports, imp), env = namespaceEnv(self))
}
namespaceImports <- function(ns) {
    env <- namespaceEnv(ns)
    if (identical(env, .BaseNamespaceEnv)) NULL
    else get(".__IMPORTS__.", env = env)
}
namespaceExport <- function(ns, vars) {
    #**** FIXME: need to worry about uniqueness of names?
    if (namespaceIsSealed(ns))
        stop("cannot add to exports of a sealed namespace")
    env <- namespaceEnv(ns)
    if (identical(env, .BaseNamespaceEnv))
        stop("cannot export from base name space")
    new <- makeImportExportNames(vars)
    undef <- new[! sapply(new, exists, env = env)]
    if (length(undef) != 0) {
        undef <- do.call("paste", as.list(c(undef, sep=", ")))
        stop(paste("undefined exports:", undef))
    }
    old <- get(".__EXPORTS__.", env = env)
    assign(".__EXPORTS__.", c(new, old), env = env)
}
namespaceExports <- function(ns) {
    env <- namespaceEnv(ns)
    if (identical(env, .BaseNamespaceEnv)) ls(NULL)
    else unique(names(get(".__EXPORTS__.", env = env)))
}
namespaceEnv <- function(ns) asNamespace(ns)
namespaceGet <- function(ns, name) {
    env <- namespaceEnv(ns)
    if (identical(env, .BaseNamespaceEnv))
        get(name, env)
    else {
        if (! namespaceIsSealed(ns))
            stop("cannot get values from unsealed namespace")
        exports <- get(".__EXPORTS__.", env = env)
        intname <- exports[name]
        if (is.na(intname))
            stop(paste(name, "is not an exported variable"))
        get(intname, env = env)
    }
}
"::" <- function(pkg,name){
    pkg <- as.character(substitute(pkg))
    name <- as.character(substitute(name))
    namespaceGet(pkg, name)
}
sealNamespace <- function(ns) {
    env <- namespaceEnv(ns)
    if (identical(env, .BaseNamespaceEnv))
        stop("cannot seal base name space yet")
    if (namespaceIsSealed(ns)) stop("already sealed")
    lockEnvironment(env, TRUE)
    lockEnvironment(parent.env(env), TRUE)
}
namespaceIsSealed <- function(ns)
   environmentIsLocked(namespaceEnv(ns))
namespaceName <- function(ns) {
    env <- namespaceEnv(ns)
    if (identical(env, .BaseNamespaceEnv)) "base"
    else get(".__NAMESPACE__.", env = env)
}
doNamespaceLibrary <- function(package, which.lib.loc, lib.loc,
                               logical.return) {
    tt <- try({
        ns <- loadNamespace(package, c(which.lib.loc, lib.loc))
        attachNamespace(ns, prefix="package")
    })
   if (inherits(tt, "try-error")) 
       if (logical.return) 
           return(FALSE)
       else stop("package/namespace load failed")
   else
       if (logical.return)
           return(TRUE)
       else
           return(invisible(.packages()))
}
.First.lib <- function(lib, pkg) {
    .Internal(registerNamespace("base", .BaseNamespaceEnv))
}
