\name{simpcond}
\alias{exception}
\alias{simple.exception}
\alias{simple.warning}
\alias{signal.condition}
\alias{with.handlers}
\alias{muffle.warnings}
\alias{ignore.errors}
\alias{abort}
\alias{find.reset}
\alias{compute.resets}
\alias{invoke.reset}
\alias{invoke.reset.interactively}
\alias{with.resets}

\title{Simple Condition Handling Mechanism}
\description{
  A simple condition handling mechanism for R.  This is in no sense
  meant as a final implementation but rather as a test bed for trying out
  ideas for how this might work.
}

\usage{
    simple.warning(message, call = NULL)
    simple.exception(message, call = NULL)
    signal.condition(c)
    try.catch(expr, ..., finally = NULL)
    with.handlers(expr, ...)
    muffle.warnings(expr)
    ignore.errors(expr)
    abort()
    find.reset(name, cond = NULL)
    compute.resets(cond = NULL)
    invoke.reset(r, ...)
    invoke.reset.interactively(r)
    with.resets(expr, ...)
}

\arguments{
  \item{string}{ string describing the exception }
  \item{call}{ a call expression }
  \item{cond}{condition object}
  \item{expr}{expression to execute for value}
  \item{finally}{expression to execute before returning or continuing a throw}
  \item{...}{\code{try.catch} and \code{with.handlers}: named
	     arguments with names representing condition classes and
             values that are functions of one argument;
	     \code{invoke.reset}: arguments for the reset handler;
	     \code{with.resets}: named arguments representing the resets
	      established.}
  \item{name}{a reset name}
  \item{r}{a reset name or a reset object}
}

\details{

   Conditions are objects inheriting from the abstract class
   \code{contition}.  Exceptions and warnings are objects inheriting
   from the abstract subclasses \code{exception} and \code{warning}.
   The class \code{simple.exception} is the class currently used by
   \code{stop} and all internal error signals.  Similarly,
   \code{simple.warning} is used by \code{warning}.  The constructors
   by the same names take a string describing the exception as
   argument and an optional call returns a \code{simple.exception}
   object.

   Conditions are signaled by \code{signal.condition}.  In addition,
   the \code{stop} and \code{warning} functions have been modified to
   also accept condition arguments.  

   The function \code{try.catch} evaluates its expression argument
   in a context where the handlers provided in the \code{...} argument
   are available.  The \code{finally} expression is then evaluated in
   the context in which \code{try.catch} was called; that is, the
   handlers supplied to the current \code{try.catch} call are not
   active when the \code{finally} expression is evaluated.

   Handlers provided in the \code{...} argument to \code{try.catch}
   are established for the duration of the evaluation of \code{expr}.
   If no condition is signaled when evaluating \code{expr} then
   \code{try.catch} returns the value of the expression.

   If a condition is signaled while evaluating \code{expr} then
   established handlers are checked, starting with the most recently
   established ones, for one matching the class of the condition.  If
   a handler is found then control is transferred to the
   \code{try.catch} call that established the handler, the handlers in
   that call are disestablished, the handler is called with the
   condition as its argument, and the result returned by the handler is
   returned as the value of the \code{try.catch} call.

   Default handlers are provided for warnings and exceptions.
   For all other contidions, an exception is signaled if no handler is
   found.

   Calling handlers are established by \code{with.handlers}.  If
   a condition is signaled and the applicable handler is a calling
   handler, then the handler is called by \code{signal.condition} in
   the context where the condition was signaled but with the available
   handlers restricted to those below the handler called in the
   handler stack.  If the handler returns, then its return value is
   returned by \code{signal.condition}.

   \code{muffle.warnings} evaluates its expression in a context that
   ignores all warnings.  \code{ignore.errors} is analogous to
   \code{try}: it evaluates the expression and, if no exception is
   raised, returns the result.  If an exception is raised then it
   returns the exception object.

   \code{abort} is used by \code{stop} to jump to the top level (or to
   a function frame where \code{restart} has been called, if there is
   one.

   Resets are used for establishing recovery protocols.  The can be
   established using \code{with.resets}.  One pre-established reset is
   an abort reset that represents a jump to top level.  The \code{abort}
   function is a convenient shorthand for invoking the abort reset.

   \code{find.reset} and \code{compute.resets} find the available resets.
   \code{find.reset} returns the most recently established reset of
   the specified name.  \code{compute.resets} returns a list of all
   resets.  Both can be given a condition argument and will then ignore
   resets that do not apply to the condition.

   \code{invoke.reset} transfers control to the point where the
   specified reset was established and calls the reset's handler with the
   arguments, if any, given as additional arguments to
   \code{invoke.reset}.
   
   New resets for \code{with.resets} can be specified in several ways.
   The simplest is in \code{name=function} form where the function is
   the handler to call when the restart is invoked.  Another simple
   variant is as \code{name=string} where the string is stored in the
   \code{message} field of the restart object returned by
   \code{find.restart}; in this case the handler ignores its arguments
   and returns NULL.  The most flexible form of a restart specification
   is as a list that can include several fields, including
   \code{hander}, \code{message}, \code{test}, and
   \code{restarts.honored}.  The \code{test} field shold contain a
   function of one argument, a condition, that returns \code{TRUE} if
   the restart applies to the condition and \code{FALSE} if it does
   not; the default function returns \code{TRUE} for all conditions.  The
   \code{restarts.honored} field is a logical value that is true if
   transfers of control to this restart should stop if they fins a
   call frame on the stack that has had \code{restart} called on it.
   The default value is \code{FALSE}, but it is \code{TRUE} for the
   default \code{abort} restart.

   One additional field that can be specified for a reset is
   \code{interactice}.  This should be a function of no arguments that
   returns a list of arguments to pass to the reset handler.  The list
   could be obtained by interacting with the user if necessary.  The
   function \code{invoke.restart.interactively} calls this function to
   obtain the arguments to use when invoking the restart.
}

\references{The \code{try.catch} mechanism is similar to Java
  exception handling.  Calling handlers are based on Common Lisp and
  Dylan.  Resets are based on the Common Lisp restart mechanism.}

\author{Luke Tierney}

\examples{
try.catch(1, finally=print("Hello"))
e<-simple.exception("test exception")
\dontrun{stop(e)}
\dontrun{try.catch(stop(e), finally=print("Hello"))}
\dontrun{try.catch(stop("fred"), finally=print("Hello"))}
try.catch(stop(e), exception = function(e) e, finally=print("Hello"))
try.catch(stop("fred"),  exception = function(e) e, finally=print("Hello"))
with.handlers({ warning("A"); 1+2 }, warning = function(w) {})
{ try(abort()); 1}
\dontrun{{ with.resets(stop("A"), abort = function() {}); 1}}
with.resets(invoke.reset(find.reset("foo"), 1, 2), foo = function(x, y) {x + y})
}
