module to-haskell where

open import lib
open import fore
open import tpctxt
open import term-to-string

kind-to-haskell : term → string
kind-to-haskell Star = "*"
kind-to-haskell (Arrow t1 t2) = (kind-to-haskell t1) ^ " -> " ^ (kind-to-haskell t2)
kind-to-haskell (Parens t) = "(" ^ (kind-to-haskell t) ^ ")"
kind-to-haskell t = "[internal error: unexpected form of term in a kind: " ^ (term-to-string t) ^ "]"

type-to-haskell : tpctxt → term → string
type-to-haskell g (Abstract Lam _ _ _) = "[internal error: translating type-level lambdas not implemented yet]"
type-to-haskell g (Abstract Forall x t1 t2) = "forall (" ^ x ^ " :: " ^ (kind-to-haskell t1) ^ ") . " ^ (type-to-haskell g t2)
type-to-haskell g (App t1 t2) = (type-to-haskell g t1) ^ " " ^ (type-to-haskell g t2)
type-to-haskell g (Arrow t1 t2) = (type-to-haskell g t1) ^ " -> " ^ (type-to-haskell g t2)
type-to-haskell g (Parens t) = "(" ^ (type-to-haskell g t) ^ ")"
type-to-haskell g (Var x) with trie-contains g x 
... | tt = x -- must be a type constructor
... | ff = x -- this must be a type variable
type-to-haskell g t = "[internal error: unexpected form of expression in a type: " ^ (term-to-string t) ^ "]"

isFoldUnfold : tpctxt → term → 𝔹
isFoldUnfold g (FoldUnfold _ _) = tt
isFoldUnfold g (Parens t) = isFoldUnfold g t
isFoldUnfold g (App t1 t2) with isFoldUnfold g t1 
... | tt = is-type g t2 
... | ff = ff
isFoldUnfold g _ = ff

term-to-haskell : (countReductions : 𝔹) → tpctxt → term → string
term-to-haskell countReductions g (Var x) = x
term-to-haskell countReductions g (App t1 t2) with is-type g t2
... | tt = term-to-haskell countReductions g t1
... | ff = 
  let doIncr = countReductions && ~ (isFoldUnfold g t1) in
   (if doIncr then "(incr " else "")
    ^ "(" ^ (term-to-haskell countReductions g t1) ^ " " ^ (term-to-haskell countReductions g t2) ^ 
    (if doIncr then "))" else ")")
term-to-haskell countReductions g (Abstract Lam x t' t) with keep (is-kind t')
... | tt , p = term-to-haskell countReductions (trie-insert g x (TpClassifier t' p)) t
... | ff , p = "(\\ " ^ x ^ " -> " ^ (term-to-haskell countReductions (trie-insert g x (Classifier t' p (Var x) (Uvar x) tt)) t) ^ ")"
term-to-haskell countReductions g (FoldUnfold x Unfold) = "unfold" ^ x
term-to-haskell countReductions g (FoldUnfold x Fold) = "Fold" ^ x
term-to-haskell countReductions g (Parens t) = "(" ^ (term-to-haskell countReductions g t) ^ ")"
term-to-haskell countReductions g t = "[internal error: term-to-haskell not implemented for " ^ (term-to-string t) ^ "]"

strip-top-type-lambdas : term → (𝕃 string) × term
strip-top-type-lambdas (Abstract Lam x _ t) with strip-top-type-lambdas t
... | vars , t' = (x :: vars) , t'
strip-top-type-lambdas t = [] , t

var-string : 𝕃 string → string
var-string vars = (string-concat-sep " " vars)

cmd-to-haskell : (countReductions : 𝔹) → tpctxt → cmd → string
cmd-to-haskell countReductions g (Define _ x _ _) with trie-lookup g x
cmd-to-haskell countReductions g (Define _ x _ _) | just (Classifier tp _ t _ _) = 
  x ^ " :: " ^ (type-to-haskell g (min-parens tp)) ^ "\n" ^ x ^ " = " ^ (term-to-haskell countReductions g (min-parens t)) ^ "\n\n"
cmd-to-haskell countReductions g (Define _ x _ _) | just (TypeDefinition ff k _ tp) with strip-top-type-lambdas tp  
... | vars , tp' = "type " ^ x ^ " " ^ (var-string vars) ^ " = " ^ (type-to-haskell g tp') ^ "\n\n"
cmd-to-haskell countReductions g (Define _ x _ _) | just (TypeDefinition tt k _ tp) with strip-top-type-lambdas tp  
... | vars , tp' = "newtype " ^ x ^ " " ^ (var-string vars) ^ " = " ^ "Fold" ^ x ^ 
                   " { unfold" ^ x ^ " :: " ^ (type-to-haskell g tp') ^ "}\n\n"
cmd-to-haskell countReductions g (Define _ x _ _) | just (TpClassifier _ _) = "[internal error: encountered TpClassifier ctxtElt]"
cmd-to-haskell countReductions g (Define _ x _ _) | nothing = "[internal error: missing definition of " ^ x ^ "\n"
cmd-to-haskell countReductions g _ = ""

cmds-to-haskellh : (countReductions : 𝔹) → tpctxt → cmds → string
cmds-to-haskellh countReductions g = cmds-map (cmd-to-haskell countReductions g)

modulenameh : 𝕃 char → 𝕃 char
modulenameh [] = []
modulenameh ('.' :: cs) = cs
modulenameh (_ :: cs) = modulenameh cs

modulename : string → string
modulename s = 𝕃char-to-string (reverse (modulenameh (reverse (string-to-𝕃char s))))

cmds-to-haskell : (countReductions : 𝔹) → (filename : string) → tpctxt → cmds → string
cmds-to-haskell countReductions filename g cs = 
  "{-# OPTIONS_GHC -XRankNTypes -XKindSignatures #-}\n" ^ 
  "module " ^ (modulename filename) ^ " where\n" ^ 
  "import System.IO.Unsafe\n" ^ 
  "import Data.IORef\n" ^
  "\n" ^
  "numReductions :: IORef Int\n" ^
  "{-# NOINLINE numReductions #-}\n" ^
  "numReductions = unsafePerformIO (newIORef 0)\n" ^
  "\n" ^
  "incr :: a -> a\n" ^
  "incr r = unsafePerformIO\n" ^
  "           (do\n" ^
  "               x <- readIORef numReductions\n" ^
  "               writeIORef numReductions (x + 1)\n" ^
  "               return r)\n" ^
  "\n" ^
  "reset :: a -> a\n" ^
  "reset r = unsafePerformIO\n" ^
  "           (do\n" ^
  "               writeIORef numReductions 0\n" ^
  "               return r)\n\n" ^
  "getNumReds :: () -> Int\n" ^
  "getNumReds () = (unsafePerformIO (readIORef numReductions))\n" ^
  (cmds-to-haskellh countReductions g cs)
