-- defines normal-order reduction for untyped lambda calculus, and related functions like renaming functions.
module untyped-lam where

open import lib
open import fore-types
open import tpctxt

-- uterm is defined in tpctxt

{- map variable names to variable names, but make sure that if we map x to y, we also map y to x -}
renamectxt : Set
renamectxt = trie string

empty-renamectxt : renamectxt
empty-renamectxt = empty-trie

renamectxt-contains : renamectxt → string → 𝔹
renamectxt-contains r s = trie-contains r s

renamectxt-insert : renamectxt → string → string → renamectxt
renamectxt-insert r s x = trie-insert r s x

eq-var : renamectxt → var → var → 𝔹
eq-var r x y with x =string y
eq-var r x y | tt = tt
eq-var r x y | ff with trie-lookup r x
eq-var r x y | ff | just x' = y =string x'
eq-var r x y | ff | nothing with trie-lookup r y
eq-var r x y | ff | nothing | just y' = x =string y'
eq-var r x y | ff | nothing | nothing = ff

pick-new-name : string → string
pick-new-name x = x ^ "'"

{- rename-away-from x g r rename the variable x to be some new name (related to x)
   which is not in the given stringset renamectxt r nor declared in the given tpctxt g. -}
{-# NO_TERMINATION_CHECK #-}
rename-away-from : string → (var → 𝔹) → renamectxt → string
rename-away-from x g r =
  if (g x) then
    rename-away-from (pick-new-name x) g r
  else if (renamectxt-contains r x) then
    rename-away-from (pick-new-name x) g r
  else x

{- subst g t' x r t:
     0. we will rename variables away from the variables for which g returns tt
     1. t' is the term being substituted
     2. x is the variable being usubstituted for
     3. r is a renaming which we will also apply as we go
     4. t is a term into which we are usubstituting t' for x -}
usubst : (var → 𝔹) → renamectxt → uterm → var → uterm → uterm
usubst g r t' x (Uvar y) with eq-var r x y 
usubst g r t' x (Uvar y) | tt = t'
usubst g r t' x (Uvar y) | ff with trie-lookup r y 
usubst g r t' x (Uvar y) | ff | just y' = Uvar y'
usubst g r t' x (Uvar y) | ff | nothing = Uvar y
usubst g r t' x (Uapp t1 t2) = Uapp (usubst g r t' x t1) (usubst g r t' x t2)
usubst g r t' x (Ulam y t2) with x =string y 
usubst g r t' x (Ulam y t2) | tt = (Ulam y t2)
usubst g r t' x (Ulam y t2) | ff = 
  if g y || renamectxt-contains r y then
    (let y' = rename-away-from y g r in
     let r' = renamectxt-insert r y y' in
      Ulam y' (usubst g r' t' x t2))
  else
    Ulam y (usubst g r t' x t2)

{-# NO_TERMINATION_CHECK #-}
{- return the whnf of the given term, together with the number of reductions we performed. -}
uwhnf : tpctxt → renamectxt → uterm → uterm × ℕ
uwhnf g r (Uapp t1 t2) with uwhnf g r t1
uwhnf g r (Uapp _ t2) | Ulam x t' , c1 with uwhnf g r (usubst (trie-contains g) r t2 x t')
uwhnf g r (Uapp _ t2) | Ulam x t' , c1 | t'' , c2 = t'' , suc (c1 + c2)
uwhnf g r (Uapp _ t2) | t1' , c1 = Uapp t1' t2 , c1
uwhnf g r (Uvar x) with trie-lookup g x
uwhnf g r (Uvar x) | just (Classifier _ _ _ d _) = uwhnf g r d
uwhnf g r (Uvar x) | _ = Uvar x , 0
uwhnf g r x = x , 0

{-# NO_TERMINATION_CHECK #-}
{- Normalize the given uterm using normal order reduction -}
unorm : tpctxt → renamectxt → uterm → uterm × ℕ
unorm-whnf : tpctxt → renamectxt → uterm → uterm × ℕ
unorm g r t with uwhnf g r t
unorm g r t | t' , c1 with unorm-whnf g r t'
unorm g r t | t' , c1 | t'' , c2 = t'' , c1 + c2
unorm-whnf g r (Uapp t1 t2) with unorm-whnf g r t1 | unorm g r t2
unorm-whnf g r (Uapp t1 t2) | t1' , c1 | t2' , c2 = Uapp t1' t2' , c1 + c2
unorm-whnf g r (Ulam x t2) with unorm g (renamectxt-insert r x x) t2
unorm-whnf g r (Ulam x t2) | t2' , c = Ulam x t2' , c
unorm-whnf g r x = x , 0

eraseh : tpctxt → term → maybe uterm
eraseh g (Var x) with trie-lookup g x
eraseh g (Var x) | just (Classifier _ _ _ _ _) = just (Uvar x)
eraseh g (Var x) | _ = nothing
eraseh g (FoldUnfold _ _) = nothing
eraseh g Star = nothing
eraseh g (Arrow _ _) = nothing
eraseh g (Abstract All _ _ _) = nothing
eraseh g (Abstract Lam x t1 t2) with keep (is-kind t1) 
eraseh g (Abstract Lam x t1 t2) | tt , p = eraseh (trie-insert g x (TpClassifier t1 p)) t2
eraseh g (Abstract Lam x t1 t2) | ff , p with eraseh (trie-insert g x (Classifier t1 p (Var x) (Uvar x) tt)) t2
eraseh g (Abstract Lam x t1 t2) | ff , p | nothing = nothing -- should not happen
eraseh g (Abstract Lam x t1 t2) | ff , p | just t2' = just (Ulam x t2')
eraseh g (Parens t) = eraseh g t
eraseh g (App t1 t2) with eraseh g t1 
eraseh g (App t1 t2) | nothing = eraseh g t2
eraseh g (App t1 t2) | just t1' with eraseh g t2
eraseh g (App t1 t2) | just t1' | nothing = just t1'
eraseh g (App t1 t2) | just t1' | just t2' = just (Uapp t1' t2')

erase : tpctxt → term → uterm
erase g t with eraseh g t
erase g t | nothing = Ulam "x" (Uvar "x")
erase g t | just t' = t'

unorm-erase : tpctxt → term → uterm × ℕ 
unorm-erase g t = unorm g empty-renamectxt (erase g t)

uterm-to-racket : (countReductions : 𝔹) → uterm → string
uterm-to-racket countReductions (Uvar x) = x
uterm-to-racket countReductions (Uapp t1 t2) =
   "(" ^ (if countReductions then "begin (set! numReductions (+ numReductions 1)) (" else "") 
    ^ (uterm-to-racket countReductions t1) ^ " " ^ (uterm-to-racket countReductions t2) ^ 
    (if countReductions then "))" else ")")
uterm-to-racket countReductions (Ulam x t) = "(lambda (" ^ x ^ ") " ^ (uterm-to-racket countReductions t) ^ ")"

