module term-to-string where

open import lib
open import fore-types

drop-parens : term → term
drop-parens Star = Star
drop-parens (Parens t) = drop-parens t
drop-parens (Var x) = Var x
drop-parens (App t1 t2) = App (drop-parens t1) (drop-parens t2)
drop-parens (Arrow t1 t2) = Arrow (drop-parens t1) (drop-parens t2)
drop-parens (Abstract b x t1 t2) = (Abstract b x (drop-parens t1) (drop-parens t2))
drop-parens (FoldUnfold x b) = FoldUnfold x b

data no-parens : term → Set where
  no-parens-Abstract : ∀{b : binder}{x : var}{t1 : term}{t2 : term} → no-parens t1 → no-parens t2 → no-parens (Abstract b x t1 t2)
  no-parens-App : ∀{x y : term} → no-parens x → no-parens y → no-parens (App x y)
  no-parens-Arrow : ∀{x y : term} → no-parens x → no-parens y → no-parens (Arrow x y)
  no-parens-Star : no-parens Star
  no-parens-Var : ∀{x : var} → no-parens (Var x)
  no-parens-FoldUnfold : ∀{x : var}{b : foldOrUnfold} → no-parens (FoldUnfold x b)

drop-parens-no-parens : ∀(t : term) → no-parens (drop-parens t)
drop-parens-no-parens Star = no-parens-Star
drop-parens-no-parens (Var _) = no-parens-Var
drop-parens-no-parens (App t1 t2) = no-parens-App (drop-parens-no-parens t1) (drop-parens-no-parens t2)
drop-parens-no-parens (Arrow t1 t2) = no-parens-Arrow (drop-parens-no-parens t1) (drop-parens-no-parens t2)
drop-parens-no-parens (Abstract b x t1 t2) = no-parens-Abstract (drop-parens-no-parens t1) (drop-parens-no-parens t2)
drop-parens-no-parens (Parens t) = drop-parens-no-parens t
drop-parens-no-parens (FoldUnfold t b) = no-parens-FoldUnfold

-- return true iff we need parens around the given term if it is the left part of an Arrow-term
min-parens-arrow-left : term → 𝔹
min-parens-arrow-left (Arrow _ _) = tt
min-parens-arrow-left (App _ (Abstract _ _ _ _)) = tt
min-parens-arrow-left (Abstract _ _ _ _) = tt
min-parens-arrow-left _ = ff

-- return true iff we need parens around the given term if it is the left part of an App-term
min-parens-app-left : term → 𝔹
min-parens-app-left (Arrow _ _) = tt
min-parens-app-left (Abstract _ _ _ _) = tt
min-parens-app-left (App _ (Abstract _ _ _ _)) = tt
min-parens-app-left _ = ff

-- return true iff we need parens around the given term if it is the right part of an App-term
min-parens-app-right : term → 𝔹
min-parens-app-right (App _ _) = tt
min-parens-app-right (Arrow _ _) = tt
min-parens-app-right _ = ff

insert-parens-if : 𝔹 → term → term
insert-parens-if tt t = Parens t
insert-parens-if ff t = t

min-parens-h : (t : term) → no-parens t → term 
min-parens-h Star _ = Star
min-parens-h (Parens t) ()
min-parens-h (Var x) _ = (Var x)
min-parens-h (FoldUnfold x b) _ = (FoldUnfold x b)
min-parens-h (Abstract b x t1 t2) (no-parens-Abstract p1 p2) = Abstract b x (min-parens-h t1 p1) (min-parens-h t2 p2)
min-parens-h (Arrow t1 t2) (no-parens-Arrow p1 p2) = Arrow (insert-parens-if (min-parens-arrow-left t1) (min-parens-h t1 p1))
                                                           (min-parens-h t2 p2)
min-parens-h (App t1 t2) (no-parens-App p1 p2) = App (insert-parens-if (min-parens-app-left t1) (min-parens-h t1 p1))
                                                     (insert-parens-if (min-parens-app-right t2) (min-parens-h t2 p2))

min-parens : term → term 
min-parens x = min-parens-h (drop-parens x) (drop-parens-no-parens x)

binder-to-string : binder → string
binder-to-string Lam = "λ"
binder-to-string All = "∀"

foldOrUnfold-to-string : foldOrUnfold → string
foldOrUnfold-to-string Fold = ""
foldOrUnfold-to-string Unfold = "! "

term-to-string-h : term → string
term-to-string-h Star = "*"
term-to-string-h (Parens t) = "(" ^ (term-to-string-h t) ^ ")"
term-to-string-h (Var x) = x
term-to-string-h (App t1 t2) = (term-to-string-h t1) ^ " " ^ (term-to-string-h t2)
term-to-string-h (Arrow t1 t2) = (term-to-string-h t1) ^ " → " ^ (term-to-string-h t2)
term-to-string-h (Abstract b x t1 t2) = (binder-to-string b) ^ " " ^ x ^ " : " ^ (term-to-string-h t1) ^ " , " ^ (term-to-string-h t2) 
term-to-string-h (FoldUnfold x b) = "[ " ^ x ^ " " ^ (foldOrUnfold-to-string b) ^ " ]"

term-to-string : term → string
term-to-string t = term-to-string-h (min-parens t)

{- A significant theorem about our min-parens algorithm: once we insert drop all parens and then
  insert parens using the min-parens function, norm-term will not modify the term -}

min-parens-h-norm-term : ∀ (t : term)(p : no-parens t) → let t' = min-parens-h t p in t' ≡ norm-term t'
min-parens-h-norm-term Star _ = refl
min-parens-h-norm-term (Parens _) ()
min-parens-h-norm-term (Var _) _ = refl
min-parens-h-norm-term (FoldUnfold _ _) _ = refl
min-parens-h-norm-term (Abstract _ _ _ _) (no-parens-Abstract _ _) = refl
min-parens-h-norm-term (Arrow (Arrow _ _) _) (no-parens-Arrow _ _) = refl
min-parens-h-norm-term (Arrow (App _ (Abstract _ _ _ _)) _) (no-parens-Arrow _ _) = refl
min-parens-h-norm-term (Arrow (Abstract _ _ _ _) _) (no-parens-Arrow _ _) = refl
min-parens-h-norm-term (Arrow (Var _) _) (no-parens-Arrow _ _) = refl
min-parens-h-norm-term (Arrow Star _) (no-parens-Arrow _ _)  = refl
min-parens-h-norm-term (Arrow (FoldUnfold _ _) _) (no-parens-Arrow _ _)  = refl
min-parens-h-norm-term (Arrow (App _ Star) _) (no-parens-Arrow (no-parens-App _ _) _) = refl
min-parens-h-norm-term (Arrow (App _ (Var _)) _) (no-parens-Arrow (no-parens-App _ _) _)  = refl
min-parens-h-norm-term (Arrow (App _ (FoldUnfold _ _)) _) (no-parens-Arrow (no-parens-App _ _) _)  = refl
min-parens-h-norm-term (Arrow (App _ (App _ _)) _) (no-parens-Arrow (no-parens-App _ _) _)  = refl
min-parens-h-norm-term (Arrow (App _ (Arrow _ _)) _) (no-parens-Arrow (no-parens-App _ _) _) = refl
min-parens-h-norm-term (Arrow (App _ (Parens _ )) _) (no-parens-Arrow (no-parens-App _ ()) _)
min-parens-h-norm-term (Arrow (Parens _ ) _) (no-parens-Arrow () _) 

min-parens-h-norm-term (App (App _ (Abstract _ _ _ _ )) Star) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Abstract _ _ _ _ )) (Var _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Abstract _ _ _ _ )) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Abstract _ _ _ _ )) (App _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Abstract _ _ _ _ )) (Arrow _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Abstract _ _ _ _ )) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ _) (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (App _ Star) Star) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ Star) (Var _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ Star) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ Star) (App _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ Star) (Arrow _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ Star) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ _) (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (App _ (Var _)) Star) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Var _)) (Var _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Var _)) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Var _)) (App _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Var _)) (Arrow _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Var _)) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ _) (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (App _ (FoldUnfold _ _)) Star) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (FoldUnfold _ _)) (Var _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (FoldUnfold _ _)) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (FoldUnfold _ _)) (App _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (FoldUnfold _ _)) (Arrow _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (FoldUnfold _ _)) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ _) (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (App _ (App _ _)) Star) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (App _ _)) (Var _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (App _ _)) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (App _ _)) (App _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (App _ _)) (Arrow _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (App _ _)) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ _) (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (App _ (Arrow _ _)) Star) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Arrow _ _)) (Var _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Arrow _ _)) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Arrow _ _)) (App _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Arrow _ _)) (Arrow _ _)) (no-parens-App (no-parens-App _ _) _) = refl
min-parens-h-norm-term (App (App _ (Arrow _ _)) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ _) (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (App _ (Parens _)) Star) (no-parens-App (no-parens-App _ ()) _) 
min-parens-h-norm-term (App (App _ (Parens _)) (Var _)) (no-parens-App (no-parens-App _ ()) _) 
min-parens-h-norm-term (App (App _ (Parens _)) (FoldUnfold _ _)) (no-parens-App (no-parens-App _ ()) _) 
min-parens-h-norm-term (App (App _ (Parens _)) (App _ _)) (no-parens-App (no-parens-App _ ()) _) 
min-parens-h-norm-term (App (App _ (Parens _)) (Parens _)) (no-parens-App (no-parens-App _ ()) _) 
min-parens-h-norm-term (App (App _ (Parens _)) (Arrow _ _)) (no-parens-App (no-parens-App _ ()) _) 
min-parens-h-norm-term (App (App _ (Parens _)) (Abstract _ _ _ _)) (no-parens-App (no-parens-App _ ()) (no-parens-Abstract _ _)) 

min-parens-h-norm-term (App Star Star) (no-parens-App _ _) = refl
min-parens-h-norm-term (App Star (Var _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App Star (FoldUnfold _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App Star (App _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App Star (Arrow _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App Star (Abstract _ _ _ _)) (no-parens-App _ (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (Var _) Star) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Var _) (Var _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Var _) (FoldUnfold _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Var _) (App _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Var _) (Arrow _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Var _) (Abstract _ _ _ _)) (no-parens-App _ (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (FoldUnfold _ _) Star) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (FoldUnfold _ _) (Var _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (FoldUnfold _ _) (FoldUnfold _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (FoldUnfold _ _) (App _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (FoldUnfold _ _) (Arrow _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (FoldUnfold _ _) (Abstract _ _ _ _)) (no-parens-App _ (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (Arrow _ _) Star) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Arrow _ _) (Var _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Arrow _ _) (FoldUnfold _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Arrow _ _) (Arrow _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Arrow _ _) (App _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Arrow _ _) (Abstract _ _ _ _)) (no-parens-App _ (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App (Abstract _ _ _ _) Star) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Abstract _ _ _ _) (Var _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Abstract _ _ _ _) (FoldUnfold _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Abstract _ _ _ _) (App _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Abstract _ _ _ _) (Arrow _ _)) (no-parens-App _ _) = refl
min-parens-h-norm-term (App (Abstract _ _ _ _) (Abstract _ _ _ _)) (no-parens-App _ (no-parens-Abstract _ _)) = refl

min-parens-h-norm-term (App _ (Parens _)) (no-parens-App _ ()) 
min-parens-h-norm-term (App (Parens _) _) (no-parens-App () _) 

min-parens-norm-term : ∀(t : term) → let t' = min-parens t in t' ≡ norm-term t'
min-parens-norm-term t = min-parens-h-norm-term (drop-parens t) (drop-parens-no-parens t)
