module my-fore-main where

import parse
open import lib
open import fore-types
import fore

open import tpctxt
open import untyped-lam
open import term-to-string
open import product-thms
open import to-haskell

module parsem = parse fore.gratr2-nt ptr
open parsem
open parsem.parse fore.rrs fore.fore-rtn
open import run ptr

infixl 8 _=binder_ 

_=binder_ : binder → binder → 𝔹
All =binder All = tt
Lam =binder Lam = tt
_ =binder _ = ff

{- subst g t' x r t:
     0. g is the type context including global and local variables (so we can rename away from them)
     1. t' is the term being substituted
     2. x is the variable being substituted for
     3. r is a renaming which we will also apply as we go
     4. t is a type into which we are substituting t' for x -}
subst : tpctxt → renamectxt → term → var → term → term
subst g r t' x (Var y) with eq-var r x y 
subst g r t' x (Var y) | tt = t'
subst g r t' x (Var y) | ff with trie-lookup r y 
subst g r t' x (Var y) | ff | just y' = Var y'
subst g r t' x (Var y) | ff | nothing = Var y
subst g r t' x (Parens t) = Parens (subst g r t' x t)
subst g r t' x (Arrow t1 t2) = Arrow (subst g r t' x t1) (subst g r t' x t2)
subst g r t' x (App t1 t2) = App (subst g r t' x t1) (subst g r t' x t2)
subst g r t' x (Abstract b y t1 t2) with x =string y 
subst g r t' x (Abstract b y t1 t2) | tt = (Abstract b y t1 t2)
subst g r t' x (Abstract b y t1 t2) | ff = 
  let t1' = subst g r t' x t1 in
    if trie-contains g y || renamectxt-contains r y then
      (let y' = rename-away-from y (trie-contains g) r in
       let r' = renamectxt-insert r y y' in
        Abstract b y' t1' (subst g r' t' x t2))
    else
      Abstract b y t1' (subst g r t' x t2)
subst g r t' x Star = Star
subst g r t' x t = t -- shouldn't happen, as we only substitute into types.

{-# NO_TERMINATION_CHECK #-}
{- return the whnf of the given type together with a boolean saying whether or not we
   did at least one beta-reduction. -}
whnf : tpctxt → renamectxt → term → term 
whnf g r (App t1 t2) with whnf g r t1
whnf g r (App _ t2) | Abstract Lam x _ t' = whnf g r (subst g r t2 x t')
whnf g r (App _ t2) | t1'  = App t1' t2 
whnf g r (Var x) with trie-lookup g x
whnf g r (Var x) | just (TypeDefinition ff _ _ d) = d
whnf g r (Var x) | _ = (Var x)
whnf g r (Parens x) = (whnf g r x)
whnf g r x = x 

{-# NO_TERMINATION_CHECK #-}
{- Normalize the given term using normal order reduction -}
norm : tpctxt → renamectxt → term → term
norm-whnf : tpctxt → renamectxt → term → term
norm g r t = norm-whnf g r (whnf g r t)
norm-whnf g r (App t1 t2) = App (norm-whnf g r t1) (norm g r t2)
norm-whnf g r (Abstract b x t1 t2) = Abstract b x (norm g r t1) (norm g (renamectxt-insert r x x) t2)
norm-whnf g r (Arrow t1 t2) = Arrow (norm g r t1) (norm g r t2)
norm-whnf g r x = x

{- check whether or not t1 and t2 beta-equal types.  We are
   not going to bother trying to show this terminates, since we are
   not worried about it failing to terminate due to buggy code, and a
   lot of work and changes to our current setup would be needed to
   show termination -}
{-# NO_TERMINATION_CHECK #-}
eq-type : tpctxt → renamectxt → (t1 : term) → (t2 : term) → 𝔹 
eq-type-whnf : tpctxt → renamectxt → (t1 : term) → (t2 : term) → 𝔹 
eq-type g r t1 t2 = eq-type-whnf g r (whnf g r t1) (whnf g r t2)
eq-type-whnf g r (Var x) (Var y) = eq-var r x y
eq-type-whnf g r (Abstract b1 x1 dom1 ran1) (Abstract b2 x2 dom2 ran2) = 
  b1 =binder b2 && eq-type g r dom1 dom2 && eq-type g (renamectxt-insert r x1 x2) ran1 ran2
eq-type-whnf g r (Arrow dom1 ran1) (Arrow dom2 ran2) = eq-type g r dom1 dom2 && eq-type g r ran1 ran2
eq-type-whnf g r (App t1a t1b) (App t2a t2b) = eq-type g r t1a t2a && eq-type g r t1b t2b
eq-type-whnf g r Star Star = tt
eq-type-whnf g r x y = ff 

-- are t1 and t2 equal kinds
eq-kind : (t1 : term) → (t2 : term) → 𝔹 
eq-kind Star Star = tt
eq-kind (Arrow t1 t2) (Arrow t1' t2') = eq-kind t1 t1' && eq-kind t2 t2'
eq-kind (Parens t1) t1' = eq-kind t1 t1'
eq-kind t1 (Parens t1') = eq-kind t1 t1'
eq-kind _ _ = ff

wrap-vars : 𝕃 term → term → term
wrap-vars (v :: vs) head = App (wrap-vars vs head) v
wrap-vars [] head = head

{- compute-foldunfold-type k p dom ran X vars
   -- k is the kind we are still processing looking for vars we should quantify, with p a proof that k is a kind
   -- dom and ran are the head of the domain and range of the function type we are trying to create
   -- X is the recursively defined type (we use it to choose names of quantified variables)
   -- n is a ℕ for helping create the next variable 
   -- vars are the quantified variables so far -}
compute-foldunfold-type-h : (k : term) → is-kind k ≡ tt → term → term → string → ℕ → 𝕃 term → term
compute-foldunfold-type-h Star p dom ran X n vars = Arrow (wrap-vars vars dom) (wrap-vars vars ran)
compute-foldunfold-type-h (Parens k) p dom ran X n vars = compute-foldunfold-type-h k p dom ran X n vars
compute-foldunfold-type-h (Arrow k1 k2) p dom ran X n vars with &&-elim{is-kind k1} p 
compute-foldunfold-type-h (Arrow k1 k2) p dom ran X n vars | (p1 , p2) = 
  let v = ("X-" ^ X ^ (ℕ-to-string n)) in
    Abstract All v k1 (compute-foldunfold-type-h k2 p2 dom ran X (suc n) ((Var v) :: vars))
compute-foldunfold-type-h (Abstract _ _ _ _) () dom ran X n vars 
compute-foldunfold-type-h (App _ _) () dom ran X n vars 
compute-foldunfold-type-h (FoldUnfold _ _) () dom ran X n vars 
compute-foldunfold-type-h (Var _) () dom ran X n vars 

compute-foldunfold-type : (k : term) → is-kind k ≡ tt → term → term → string → term
compute-foldunfold-type k p dom ran X = compute-foldunfold-type-h k p dom ran X 0 []

data pol : Set where
  pos : pol
  neg : pol
  notfree : pol -- for saying that the variable does not occur at all

pol-neg : pol → pol 
pol-neg pos = neg
pol-neg neg = pos
pol-neg notfree = notfree

only-polarity : string → term → pol → 𝔹
only-polarity x (Var y) p with x =string y
only-polarity x (Var y) p | ff = tt
only-polarity x (Var y) pos | tt = tt
only-polarity x (Var y) neg | tt = ff
only-polarity x (Var y) notfree | tt = ff
only-polarity x (App t1 t2) p = only-polarity x t1 p && only-polarity x t2 notfree -- have to insist x is not free in type argument t2
only-polarity x (Abstract b y t1 t2) p with x =string y
only-polarity x (Abstract b y t1 t2) p | tt = tt
only-polarity x (Abstract b y t1 t2) p | ff = only-polarity x t2 p -- t1 must be a kind so we don't need to check it
only-polarity x (Arrow t1 t2) p = only-polarity x t1 (pol-neg p) && only-polarity x t2 p
only-polarity x (Parens t) p = only-polarity x t p
only-polarity x _ p = ff -- should not happen

classify : tpctxt → (t : term) → term {- the classifier of t -}
                               ⊎ string {- error message if any -}
classify g (Var x) with trie-lookup g x 
classify g (Var x) | nothing = inj₂ ("Variable " ^ x ^ " is undefined.")
classify g (Var x) | just (Classifier tp _ _ _ _) = inj₁ tp
classify g (Var x) | just (TpClassifier tp _) = inj₁ tp
classify g (Var x) | just (TypeDefinition _ k _ tp) = inj₁ k
classify g (Abstract Lam x t1 t2) with keep (is-kind t1) 
classify g (Abstract Lam x t1 t2) | tt , p with classify (trie-insert g x (TpClassifier t1 p)) t2 
classify g (Abstract Lam x t1 t2) | tt , p | inj₁ c with is-kind c
classify g (Abstract Lam x t1 t2) | tt , p | inj₁ c | tt = inj₁ (Arrow t1 c) 
classify g (Abstract Lam x t1 t2) | tt , p | inj₁ c | ff = inj₁ (Abstract All x t1 c)
classify g (Abstract Lam x t1 t2) | tt , p | inj₂ e = inj₂ e 
classify g (Abstract Lam x t1 t2) | ff , p with classify (trie-insert g x (Classifier t1 p (Var x) (Uvar x) tt)) t2 
classify g (Abstract Lam x t1 t2) | ff , p | inj₁ c with is-kind c
classify g (Abstract Lam x t1 t2) | ff , p | inj₁ c | tt =
  inj₂ ("Lambda-abstracting terms over types is not allowed in this type theory.\n" ^
        "1. The term: " ^ (term-to-string (Abstract Lam x t1 t2) ) ^ "")
classify g (Abstract Lam x t1 t2) | ff , p | inj₁ c | ff with classify g t1 
classify g (Abstract Lam x t1 t2) | ff , p | inj₁ c | ff | inj₂ e = inj₂ e
classify g (Abstract Lam x t1 t2) | ff , p | inj₁ c | ff | inj₁ Star = inj₁ (Arrow t1 c)
classify g (Abstract Lam x t1 t2) | ff , p | inj₁ c | ff | inj₁ c1 = 
  inj₂ ("A lambda-abstraction has a range which is a term, but the domain is neither a kind nor a type.\n" ^
        "1. The lambda-abstraction: " ^ (term-to-string (Abstract Lam x t1 t2)) ^ "\n" ^
        "2. The classifier of the domain: " ^ (term-to-string c1) ^ "\n" ^
        "3. The classifier of the range: " ^ (term-to-string c) ^ "")
classify g (Abstract Lam x t1 t2) | ff , p | inj₂ e = inj₂ e 
classify g (Abstract All x t1 t2) with keep (is-kind t1)
classify g (Abstract All x t1 t2) | ff , p =
   inj₂ ("The declared classifier for universally quantified variable " ^ x ^ " is not a kind:\n1. the declared classifier: " ^
         (term-to-string t1))
classify g (Abstract All x t1 t2) | tt , p with classify (trie-insert g x (TpClassifier t1 p)) t2
classify g (Abstract All x t1 t2) | tt , p | inj₂ e = inj₂ e
classify g (Abstract All x t1 t2) | tt , p | inj₁ Star = inj₁ Star
classify g (Abstract All x t1 t2) | tt , p | inj₁ c = inj₂ "FIXME"
classify g (App t1 t2) with classify g t1 | classify g t2
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 with whnf g empty-renamectxt c1
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | Arrow c1a c2a with eq-type g empty-renamectxt c1a c2
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | Arrow c1a c2a | tt = inj₁ c2a
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | Arrow c1a c2a | ff = 
  inj₂ ("The type of the argument does not match the domain type of the function in an application." ^
        "\n1. the function: " ^ (term-to-string t1) ^
        "\n2. the argument: " ^ (term-to-string t2) ^
        "\n3. the domain type of the function: " ^ (term-to-string c1a) ^
        "\n4. the type of the argument: " ^ (term-to-string c2))
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | Abstract All x c1a c2a with eq-kind c1a c2
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | Abstract All x c1a c2a | tt = inj₁ (subst g empty-renamectxt t2 x c2a)
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | Abstract All x c1a c2a | ff = 
  inj₂ ("An argument with a kind is expected, but the classifier of the argument is not an equal kind." ^
        "\n1. the function: " ^ (term-to-string t1) ^
        "\n2. the argument: " ^ (term-to-string t2) ^
        "\n3. the domain kind of the function: " ^ (term-to-string c1a) ^
        "\n4. the classifier of the argument: " ^ (term-to-string c2))
classify g (App t1 t2) | inj₁ c1 | inj₁ c2 | c1' = 
  inj₂ ("The term in functional position in an application does not have an arrow or a universal type." ^
        "\n1. The term in functional position: " ^ (term-to-string t1) ^
        "\n2. Its classifier " ^ (term-to-string c1))
classify g (App t1 t2) | inj₂ c1 | _ = inj₂ c1
classify g (App t1 t2) | _ | inj₂ c2 = inj₂ c2
classify g (Parens t) = classify g t
classify g (FoldUnfold x u) with trie-lookup g x
classify g (FoldUnfold x u) | nothing = 
  inj₂ ("In a fold/unfold term, symbol " ^ x ^ " is undefined.")
classify g (FoldUnfold x u) | just (TypeDefinition ff _ _ _) = 
  inj₂ ("In a fold/unfold term, symbol " ^ x ^ " is not a recursively defined type.")
classify g (FoldUnfold x Fold) | just (TypeDefinition tt k p d ) = 
  inj₁ (compute-foldunfold-type k p d (Var x) x)
classify g (FoldUnfold x Unfold) | just (TypeDefinition tt k p d ) = 
  inj₁ (compute-foldunfold-type k p (Var x) d x)
classify g (FoldUnfold x u) | just _ = 
  inj₂ ("In a fold/unfold term, symbol " ^ x ^ " is not a defined type.")
classify g (Arrow t1 t2) with classify g t1 | classify g t2
classify g (Arrow t1 t2) | inj₂ e | _ = inj₂ e
classify g (Arrow t1 t2) | _ | inj₂ e = inj₂ e
classify g (Arrow t1 t2) | inj₁ Star | inj₁ Star = inj₁ Star 
classify g (Arrow t1 t2) | inj₁ c1 | inj₁ Star = 
  inj₂ ("The domain type of an arrow type is not a type."
    ^ "\n1. the domain type: " ^ (term-to-string t1)
    ^ "\n2. its classifier " ^ (term-to-string c1))
classify g (Arrow t1 t2) | inj₁ _ | inj₁ c2 = 
  inj₂ ("The range type of an arrow type is not a type."
    ^ "\n1. the range type: " ^ (term-to-string t2)
    ^ "\n2. its classifier " ^ (term-to-string c2))
classify g Star = inj₂ "The kind * is being used where a type is expected."

process-command : tpctxt → cmd → (tpctxt {- the updated tpctxt unless error -} × string {- any output for the command -})
                                     ⊎ string {- error message if any -}
{- Define x tp trm means that x has classifier tp and is defined to be trm -}
process-command g (Define Plain x otp trm) with classify g trm 
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' {- computed classifier of trm -} with keep (is-kind tp)
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' | tt , p with eq-kind tp tp'
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' | tt , p | tt = 
  inj₁ (trie-insert g x (TypeDefinition ff tp p (whnf g empty-renamectxt trm)) , "")
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' | tt , p | ff = 
  inj₂ ("The declared classifier of " ^ x ^ " does not match the computed kind.\n" ^
  "1. the declared classifier is " ^ (term-to-string tp) ^ "\n" ^
  "2. the computed kind is " ^ (term-to-string tp') ^ "")
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' | ff , p with eq-type g empty-renamectxt tp tp'
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' | ff , p | tt =
  inj₁ (trie-insert g x (Classifier tp p trm (erase g trm) ff) , "")
process-command g (Define Plain x (SomeClass tp) trm) | inj₁ tp' | ff , p | ff =
  inj₂ ("The declared classifier of " ^ x ^ " does not match the computed type.\n" ^
  "1. the declared classifier is " ^ (term-to-string tp) ^ "\n" ^
  "2. the computed type is " ^ (term-to-string tp') ^ "")
process-command g (Define Plain x NoClass trm) | inj₁ tp with keep (is-kind tp) 
process-command g (Define Plain x NoClass trm) | inj₁ tp | tt , p =
  inj₁ (trie-insert g x (TypeDefinition ff tp p (whnf g empty-renamectxt trm)) , 
        (x ^ " : " ^ (term-to-string tp) ^ "\n"))
process-command g (Define Plain x NoClass trm) | inj₁ tp | ff , p =
  inj₁ (trie-insert g x (Classifier tp p trm (erase g trm) ff) , (x ^ " : " ^ (term-to-string tp) ^ "\n"))
process-command g (Define Plain x tp trm) | inj₂ errorMessage =
 inj₂ ("While checking the definition of " ^ x ^ ", this error was encountered:\n" ^ errorMessage)
process-command g (Define Rec x NoClass trm) = 
  inj₂ ("The definition of " ^ x ^ " is marked recursive, but the definition does not have a declared kind.")
process-command g (Define Rec x (SomeClass k) trm) with keep (is-kind k)
process-command g (Define Rec x (SomeClass k) trm) | ff , _ = 
  inj₂ ("The definition of " ^ x ^ " is marked recursive, but the declared classifier is not a kind."
       ^ "\n1. The declared classifier: " ^ (term-to-string k) ^ "")
process-command g (Define Rec x (SomeClass k) trm) | tt , p with classify (trie-insert g x (TpClassifier k p)) trm 
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₂ e = 
  inj₂ ("While checking the definition of " ^ x ^ ", this error was encountered:\n" ^ e)
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₁ k' with eq-kind k k'
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₁ k' | ff =
  inj₂ ("Recursively defined " ^ x ^ " is declared to have one kind, but a different classifier was computed for its definition."
    ^ "\n1. the declared kind: " ^ (term-to-string k) 
    ^ "\n2. the computed classifier: " ^ (term-to-string k'))
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₁ k' | tt with norm g empty-renamectxt trm 
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₁ k' | tt | trm' with only-polarity x trm' pos
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₁ k' | tt | trm' | ff =
  inj₂ ("The recursive definition of type \"" ^ x ^ "\" uses \"" ^ x ^ "\" in a negative position.\n" 
      ^ "1. the normalized defining term: " ^ (term-to-string trm))
process-command g (Define Rec x (SomeClass k) trm) | tt , p | inj₁ k' | tt | trm' | tt =
  inj₁ (trie-insert g x (TypeDefinition tt k p trm) , "")
process-command g (Norm x) with trie-lookup g x
process-command g (Norm x) | just (Classifier _ _ _ u _) with unorm g empty-renamectxt u 
process-command g (Norm x) | just (Classifier _ _ _ u _) | u' , c =
    inj₁ (g , x ^ " →! " ^ (uterm-to-racket ff u') ^ " (size = " ^ (ℕ-to-string (size u')) ^ ") in " ^ (ℕ-to-string c) ^ " steps\n")
process-command g (Norm x) | just (TypeDefinition ff _ _ trm) =
    inj₁ (g , x ^ " →! " ^ (term-to-string (norm g empty-renamectxt trm)) ^ "\n")
process-command g (Norm x) | _ = inj₂ ("The symbol " ^ x ^ " is not a defined term and hence cannot be normalized.")
process-command g Quit = inj₁ (g , "Quitting.\n")

process-commands : tpctxt → cmds → tpctxt × string 
process-commands g (CmdsNext Quit cs) = g , "Quitting.\n"
process-commands g (CmdsNext c cs) with process-command g c
process-commands g (CmdsNext c cs) | inj₁ (g' , msg) with process-commands g' cs
process-commands g (CmdsNext c cs) | inj₁ (g' , msg) | g'' , msg' = g'' , msg ^ msg'
process-commands g (CmdsNext c cs) | inj₂ errorMessage = g , errorMessage
process-commands g (CmdsStart c) with process-command g c
process-commands g (CmdsStart c) | inj₁ p = p
process-commands g (CmdsStart c) | inj₂ errorMessage = g , errorMessage 

cmd-to-racket : (countReductions : 𝔹) → tpctxt → cmd → string
cmd-to-racket countReductions g (Define Plain x _ _) with trie-lookup g x
cmd-to-racket countReductions g (Define Plain x _ _) | just (Classifier _ _ _ u _) = 
  "(define " ^ x ^ " " ^ (uterm-to-racket countReductions u) ^ ")\n(provide " ^ x ^ ")\n"
cmd-to-racket countReductions g (Define Plain x _ _) | just _ = ""
cmd-to-racket countReductions g (Define Plain x _ _) | nothing = "[internal error: missing definition of " ^ x ^ "\n"
cmd-to-racket countReductions g _ = ""

cmds-to-racketh : (countReductions : 𝔹) → tpctxt → cmds → string
cmds-to-racketh countReductions g = cmds-map (cmd-to-racket countReductions g)

cmds-to-racket : (countReductions : 𝔹) → tpctxt → cmds → string
cmds-to-racket countReductions g cs = 
  "#lang racket\n\n(define numReductions 0)\n(provide numReductions)\n\n" 
  ^ "(define (reset) (set! numReductions 0))\n(provide reset)\n\n"
  ^ (cmds-to-racketh countReductions g cs)

-- the first string returned is to print to stdout; the second is to print to a Racket file; the third is for Haskell
process-start : (filename : string) → (countReductions : 𝔹) → start → string × string × string
process-start filename countReductions (Cmds cmds) with process-commands empty-trie cmds
process-start filename countReductions (Cmds cmds) | g , s =
  s , cmds-to-racket countReductions g cmds , cmds-to-haskell countReductions filename g cmds 

process : (filename : string) → (countReductions : 𝔹) → {lc : 𝕃 char} → Run lc → string × string × string
process filename countReductions (ParseTree{s = "start"}{parsed-start p} ipt ::' []') = process-start filename countReductions p
process filename countReductions r = "Parsing failure (run with -" ^ "-showParsed).\n" , "" , ""

putStrRunIf : {lc : 𝕃 char} → 𝔹 → Run lc → IO ⊤
putStrRunIf tt r = putStr (Run-to-string r) >> putStr "\n"
putStrRunIf ff r = return triv

filenameh : (rsuffix : 𝕃 char) → 𝕃 char → 𝕃 char
filenameh rsuffix [] = []
filenameh rsuffix ('.' :: cs) = rsuffix ++ ('.' :: cs)
filenameh rsuffix (_ :: cs) = filenameh rsuffix cs

filename : (suffix : string) → string → string
filename suffix s = 𝕃char-to-string (reverse (filenameh (reverse (string-to-𝕃char suffix)) (reverse (string-to-𝕃char s))))

processArgs : (showRun : 𝔹) → (showParsed : 𝔹) → (countReductions : 𝔹) → 𝕃 string → IO ⊤ 
processArgs showRun showParsed countReductions (f :: []) = (readFiniteFile f) >>= processText
  where processText : string → IO ⊤
        processText x with runRtn (string-to-𝕃char x)
        processText x | s with s
        processText x | s | inj₁ cs = putStr "Characters left before failure : " >> putStr (𝕃char-to-string cs) >> putStr "\nCannot proceed to parsing.\n"
        processText x | s | inj₂ r with putStrRunIf showRun r | rewriteRun r
        processText x | s | inj₂ r | sr | r' with putStrRunIf showParsed r' | process f countReductions r'
        processText x | s | inj₂ r | sr | r' | sr' | s1 , s2 , s3 = 
          sr >> sr' >> putStr s1 >> putStr "\n" >> writeFile (filename "rkt" f) s2 >> writeFile (filename "hs" f) s3
                                     
processArgs showRun showParsed countReductions ("--showRun" :: xs) = processArgs tt showParsed countReductions xs 
processArgs showRun showParsed countReductions ("--showParsed" :: xs) = processArgs showRun tt countReductions xs 
processArgs showRun showParsed countReductions ("--countReductions" :: xs) = processArgs showRun showParsed tt xs 
processArgs showRun showParsed countReductions (x :: xs) = putStr ("Unknown option " ^ x ^ "\n")
processArgs showRun showParsed countReductions [] = putStr "Please run with the name of a file to process.\n"

main : IO ⊤
main = getArgs >>= processArgs ff ff ff

