module domains-examples where

open import Level hiding (suc ; _⊔_)
open import Relation.Binary.PropositionalEquality
open import Data.Nat hiding (_≤_)
open import Data.Bool
open import Data.Sum
open import Data.Product
open import Data.Empty

-- local imports
open import domains
open import util

open top Level.zero

{- a non-constructive principle sufficient for proving that ℕω is a predomain.
   This seems very similar to the so-called limited principle of omniscience. -}
postulate lpo : ∀ {p : poset}(c : ω-chain p) →
                  let open poset p in let open ω-chain c in
                     -- eventually constant 
                    (∃ λ (n : ℕ) → ∀ (m : ℕ) → n ≤ℕ m ≡ true → f n ≡ f m) 
                     -- always eventually strictly increasing
                  ⊎ (∀ (n : ℕ) → ∃ λ (m : ℕ) → n ≤ℕ m ≡ true × f n ≢ f m) 


ℕ-poset : poset
ℕ-poset = record { X = ℕ ; 
                   _⊑_ = λ x y → x ≤ℕ y ≡ true {- from util.agda -};
                   reflexivity = ≤ℕ-refl;
                   transitivity = ≤ℕ-trans;
                   antisymmetry = ≤ℕ-antisymm }

{- ℕω, the natural numbers extended with positive infinity,
   forms a domain.

   First we will define the datatype ℕω and the ordering ≤ℕω
   on it.

   Next, we show this is a poset.

   The hardest step comes next, where we show it is a predomain. -}

data ℕω : Set where
  +_ : ℕ → ℕω
  ω : ℕω

infixl 4 _≤ℕω_

_≤ℕω_ : ℕω → ℕω → Bool
x ≤ℕω ω = true
+ x ≤ℕω + y = x ≤ℕ y
ω ≤ℕω + y = false

≤ℕω-total : ∀ (n m : ℕω) → n ≤ℕω m ≡ true ⊎ m ≤ℕω n ≡ true
≤ℕω-total n ω = inj₁ refl
≤ℕω-total ω m = inj₂ refl
≤ℕω-total (+ x) (+ y) with ≤ℕ-total x y 
≤ℕω-total (+ x) (+ y) | inj₁ p = inj₁ p
≤ℕω-total (+ x) (+ y) | inj₂ p = inj₂ (≤ℕ-suc-left y x p) 

≤ℕω-total-nat : ∀ (n : ℕω)(m : ℕ) → n ≤ℕω + m ≡ true ⊎ + suc m ≤ℕω n ≡ true
≤ℕω-total-nat ω _ = inj₂ refl
≤ℕω-total-nat (+ x) m = ≤ℕ-total x m

≤ℕω-refl : ∀ (x : ℕω) → x ≤ℕω x ≡ true
≤ℕω-refl (+ n) = ≤ℕ-refl n
≤ℕω-refl ω = refl

≤ℕω-omega : ∀ (x : ℕω) → ω ≤ℕω x ≡ true → x ≡ ω
≤ℕω-omega ω p = refl
≤ℕω-omega (+ x) ()

≤ℕω-zero : ∀ ( x : ℕω) → x ≤ℕω + 0 ≡ true → x ≡ + 0
≤ℕω-zero (+ n) p rewrite ≤ℕ-zero n p = refl
≤ℕω-zero ω ()

≤ℕω-trans : ∀ (x y z : ℕω) → x ≤ℕω y ≡ true → y ≤ℕω z ≡ true → x ≤ℕω z ≡ true
≤ℕω-trans x y ω p1 p2 = refl
≤ℕω-trans x ω z p1 p2 rewrite ≤ℕω-omega z p2 = refl
≤ℕω-trans ω y z p1 p2 rewrite ≤ℕω-omega y p1 = p2
≤ℕω-trans (+ x) (+ y) (+ z) p1 p2 = ≤ℕ-trans x y z p1 p2

≤ℕω-antisymm : ∀ (x y : ℕω) → x ≤ℕω y ≡ true → y ≤ℕω x ≡ true → x ≡ y
≤ℕω-antisymm ω y p1 p2 rewrite ≤ℕω-omega y p1 = refl
≤ℕω-antisymm x ω p1 p2 rewrite ≤ℕω-omega x p2 = refl
≤ℕω-antisymm (+ n) (+ m) p1 p2 rewrite ≤ℕ-antisymm n m p1 p2 = refl

≤ℕω-suc-right : ∀ (a : ℕω) (x : ℕ) → a ≤ℕω (+ x) ≡ true → a ≤ℕω (+ suc x) ≡ true
≤ℕω-suc-right a x p = ≤ℕω-trans a (+ x) (+ suc x) p (≤ℕ-suc x)

ℕω-poset : poset
ℕω-poset = record { X = ℕω ; 
                   _⊑_ = λ x y → x ≤ℕω y ≡ true ;
                   reflexivity = ≤ℕω-refl ;
                   transitivity = ≤ℕω-trans;
                   antisymmetry = ≤ℕω-antisymm }

{- in this module, we prove a lemma, also called ℕω-cardinality,
   which derives a contradiction from the assumptions that
      1. the range of an ω-chain with function f is bounded, and
      2. f is strictly increasing, in the sense of the second
         option in the lpo axiom above ("always eventually
         different") 
   This lemma is used to establish one case of the ⊔ℕω-least 
   lemma, proved after this module below. -}
module ℕω-cardinality where

  {- one of the key ideas of the proof is to modify the function
     f of the ω-chain by artificially forcing it to be bounded by
     some element a of ℕω.  The function ℕω-bound takes f and a,
     and returns the artificially bounded function. -}
  ℕω-bound : (ℕ → ℕω) → ℕω → (ℕ → ℕω)
  ℕω-bound f a n = if f n ≤ℕω a then f n else a

  {- the range of the bounded function really is bounded by a -}
  ℕω-bound-bounded : ∀ (f : ℕ → ℕω) (a : ℕω)(n : ℕ) → (ℕω-bound f a n) ≤ℕω a ≡ true
  ℕω-bound-bounded f a n with Bool-dec (f n ≤ℕω a)
  ℕω-bound-bounded f a n | inj₁ p rewrite p = p
  ℕω-bound-bounded f a n | inj₂ p rewrite p = ≤ℕω-refl a
    
  {- the bounded function is still increasing -}
  ℕω-bound-increasing : ∀ (f : ℕ → ℕω)(a : ℕω) (inc : Increasing f (λ x y → x ≤ℕω y ≡ true)) →
                          Increasing (ℕω-bound f a) (λ x y → x ≤ℕω y ≡ true)
  ℕω-bound-increasing f a inc n with Bool-dec (f n ≤ℕω a) | Bool-dec (f (suc n) ≤ℕω a)
  ℕω-bound-increasing f a inc n | inj₁ p | inj₁ p' rewrite p | p' = inc n
  ℕω-bound-increasing f a inc n | inj₁ p | inj₂ p' rewrite p | p' = p
  ℕω-bound-increasing f a inc n | inj₂ p | inj₁ p' rewrite p | p' with ≤ℕω-total (f n) a 
  ℕω-bound-increasing f a inc n | inj₂ p | inj₁ p' | inj₁ p'' rewrite p'' = ⊥-elim (Bool-contr p)
  ℕω-bound-increasing f a inc n | inj₂ p | inj₁ p' | inj₂ p'' = ≤ℕω-trans a (f n) (f (suc n)) p'' (inc n)
  ℕω-bound-increasing f a inc n | inj₂ p | inj₂ p' rewrite p | p' = ≤ℕω-refl a

  {- helper lemma for ℕω-bound-strict.  I had trouble coaxing Agda (2.3.0.1) into accepting the
     lemma in one go. -}
  ℕω-bound-strict-helper2 : ∀ (f : ℕ → ℕω) (x : ℕ)
                              (inc : Increasing f (λ x y → x ≤ℕω y ≡ true))
                              (bnd : ∀ n → f n ≤ℕω + suc x ≡ true)
                              (strict : ∀ n → ∃ λ m → (n ≤ℕ m ≡ true) × (f n ≢ f m)) →
                              (n : ℕ)
                              (m : ℕ)
                              (p : f n ≢ f m) → 
                              ℕω-bound f (+ x) n ≢ ℕω-bound f (+ x) m
  ℕω-bound-strict-helper2 f x inc bnd strict n m p with ≤ℕω-total-nat (f n) x 
  ℕω-bound-strict-helper2 f x inc bnd strict n m p | inj₁ u rewrite u with ≤ℕω-total-nat (f m) x 
  ℕω-bound-strict-helper2 f x inc bnd strict n m p | inj₁ u | inj₁ u' rewrite u' = p
  ℕω-bound-strict-helper2 f x inc bnd strict n m p | inj₁ u | inj₂ u' with strict m 
  ℕω-bound-strict-helper2 f x inc bnd strict n m p | inj₁ u | inj₂ u' | ( m' , ( p1 , p2)) = 
    ⊥-elim 
      (p2 (poset-cycles-trivial ℕω-poset (+ suc x) (f m) (f m') 
             u'
             (ω-chain-mono ℕω-poset record { f = f ; increasing  = inc } m m' p1)
             (bnd m')))
  ℕω-bound-strict-helper2 f x inc bnd strict n m p | inj₂ u with strict n
  ℕω-bound-strict-helper2 f x inc bnd strict n m p | inj₂ u | ( m' , ( p1 , p2)) = 
    ⊥-elim
      (p2 (poset-cycles-trivial ℕω-poset (+ suc x) (f n) (f m') 
             u 
             (ω-chain-mono ℕω-poset record { f = f ; increasing  = inc } n m' p1)
             (bnd m')))

  {- the bounded function satisfies the "eventually always different" property
     of the second branch of lpo above. -}
  ℕω-bound-strict : ∀ (f : ℕ → ℕω) (x : ℕ)
                      (inc : Increasing f (λ x y → x ≤ℕω y ≡ true))
                      (bnd : ∀ n → f n ≤ℕω + suc x ≡ true)
                      (strict : ∀ n → ∃ λ m → (n ≤ℕ m ≡ true) × (f n ≢ f m)) →
                      ∀ n → ∃ λ m → (n ≤ℕ m ≡ true) × (ℕω-bound f (+ x) n ≢ ℕω-bound f (+ x) m)
  ℕω-bound-strict f x inc bnd strict n with strict n
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) with ≤ℕω-total-nat (f n) x 
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) | inj₁ u rewrite u with ≤ℕω-total-nat (f m) x 
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) | inj₁ u | inj₁ u' rewrite u' = 
    (m , (p1 , λ e → p2 (trans e (sym (cong1 (λ q → (if q then f m else + x)) (sym u'))))))
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) | inj₁ u | inj₂ u' with strict m 
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) | inj₁ u | inj₂ u' | ( m' , ( p1' , p2')) = 
    ⊥-elim 
      (p2' (poset-cycles-trivial ℕω-poset (+ suc x) (f m) (f m') 
             u'
             (ω-chain-mono ℕω-poset record { f = f ; increasing  = inc } m m' p1')
             (bnd m')))
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) | inj₂ u with strict n
  ℕω-bound-strict f x inc bnd strict n | ( m , ( p1 , p2 )) | inj₂ u | ( m' , ( p1' , p2')) = 
    ⊥-elim
      (p2' (poset-cycles-trivial ℕω-poset (+ suc x) (f n) (f m') 
             u 
             (ω-chain-mono ℕω-poset record { f = f ; increasing  = inc } n m' p1')
             (bnd m')))

  {- The main lemma of this module. If f is strictly increasing and also range-bounded, 
     then we can derive a contractiction by induction on the bound x. -}
  ℕω-cardinality : ∀ (f : ℕ → ℕω)(x : ℕ)(inc : Increasing f (λ x y → x ≤ℕω y ≡ true)) 
                     (bnd : ∀ n → f n ≤ℕω + x ≡ true)
                     (strict : ∀ n → ∃ λ m → (n ≤ℕ m ≡ true) × (f n ≢ f m)) → ⊥
  ℕω-cardinality f 0 inc bnd strict with strict 0 
  ℕω-cardinality f 0 inc bnd strict | ( m , ( _ , p2 )) rewrite ≤ℕω-zero (f m) (bnd m) | ≤ℕω-zero (f 0) (bnd 0) = ⊥-elim (p2 refl)
  ℕω-cardinality f (suc x) inc bnd strict = 
    ℕω-cardinality (ℕω-bound f (+ x)) x  
     (ℕω-bound-increasing f (+ x) inc)
     (ℕω-bound-bounded f (+ x))
     (ℕω-bound-strict f x inc bnd strict)

{- calculate the least upper bound of an ω-chain in ℕω,
   with the help of the essentially nonconstructive 
   postulate lpo. -}
⊔ℕω_ : ω-chain ℕω-poset → ℕω
⊔ℕω c with lpo c 
⊔ℕω c | inj₁ ( n , p ) = let open ω-chain c in f n
⊔ℕω c | inj₂ p = ω
          
-- helper function for the following lemma
⊔ℕω-upper-case1 :
  ∀ (c : ω-chain ℕω-poset) → 
  ∀ (n : ℕ) (u : ∀ (m : ℕ) → n ≤ℕ m ≡ true → ω-chain.f c n ≡ ω-chain.f c m) → 
  ∀ (n' : ℕ) → ω-chain.f c n' ≤ℕω ω-chain.f c n ≡ true
⊔ℕω-upper-case1 c n u n' with ≤ℕ-total n n' 
⊔ℕω-upper-case1 c n u n' | inj₁ u' rewrite u n' u' = ≤ℕω-refl (ω-chain.f c n')
⊔ℕω-upper-case1 c n u n' | inj₂ u' = ω-chain-mono ℕω-poset c n' n (≤ℕ-suc-left n' n u')

{- the element computed by ⊔ℕω is really an upper bound -}
⊔ℕω-upper : ∀ (c : ω-chain ℕω-poset) → upperbound (⊔ℕω c) c
⊔ℕω-upper c with lpo c 
⊔ℕω-upper c | inj₁ ( n , p ) = ⊔ℕω-upper-case1 c n p 
⊔ℕω-upper c | inj₂ p = λ n → refl

{- the element computed by ⊔ℕω is less than or equal to any upper bound -}
⊔ℕω-least : ∀ (c : ω-chain ℕω-poset) → let open ω-chain c in 
            ∀ (x : ℕω) → upperbound x c → (⊔ℕω c) ⊑ x
⊔ℕω-least c x u with lpo c
⊔ℕω-least c x u | inj₁ ( n , p ) = u n 
⊔ℕω-least c ω u | inj₂ p = refl
⊔ℕω-least c (+ x) u | inj₂ p = let open ω-chain c in 
                                 ⊥-elim
                                   (ℕω-cardinality.ℕω-cardinality f x increasing u p)

ℕω-predomain : predomain
ℕω-predomain = record { p = ℕω-poset ;
                        ⊔_ = ⊔ℕω_ ;
                        ⊔-upper = ⊔ℕω-upper ;
                        ⊔-least = ⊔ℕω-least }

ℕω-domain : domain
ℕω-domain = record { pd = ℕω-predomain ;
                     ⊥ = + 0 ;
                     ⊥-least = lem }
  where lem : ∀ (n : ℕω) → + 0 ≤ℕω n ≡ true
        lem (+ x) = refl
        lem ω = refl