package primes.trusted;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

/*
 * SetManager
 * Manages facts about sets, including equality and unions. 
 *
 * @author Ian Wehrman <iwehrman@cse.wustl.edu>
 */

public final class SetManager {

    public static final class Eq {
        final Set a;
        final Set b;
        
        private Eq(Set a, Set b) {
            this.a = a;
            this.b = b;
            runtimeCheck(a, b);
        }

        public void runtimeCheck(Set a, Set b) {
            if (a == null || b == null || !a.equals(b) || !b.equals(a)) {
                throw new RuntimeException("Equality: inequal objects");
            }

            if (this.a != a || this.b != b) {
                throw new RuntimeException("Equality: proof/object mismatch");
            }
        }
    }

    public static class EqBundle {
        public final Set a; 
        public final Set b;
        public final Eq proof;
        public EqBundle(final Set a, final Set b,
                        final Eq proof) {
            this.a = a; 
            this.b = b; 
            this.proof = proof;
        }
    }

    public static final Set NIL = new HashSet();

    public static EqBundle refl(final Set a) {
        return new EqBundle(a, a, new Eq(a, a));
    }

    public static EqBundle symm(final Set a, 
                                final Set b, 
                                Eq proof) // <a,b>
    {
        proof.runtimeCheck(a, b);
        return new EqBundle(b, a, new Eq(b, a));
    }

    public static EqBundle trans(final Set a, 
                                 final Set b, 
                                 final Set c, 
                                 Eq abProof, 
                                 Eq bcProof) {
        abProof.runtimeCheck(a, b);
        bcProof.runtimeCheck(b, c);
        return new EqBundle(a, c, new Eq(a, c));
    }

    public static final class IsUnion {
        final Set a, b, aub;

        private IsUnion(Set a, Set b, Set aub)
        {
            this.a = a;
            this.b = b;
            this.aub = aub;
            runtimeCheck(a, b, aub);
        }

        public void runtimeCheck(Set a, Set b, Set aub) {
            if (a == null || b == null || aub == null) {
                throw new RuntimeException("Union: null sets");
            }
            
            Object x;
            for (Iterator i = aub.iterator(); i.hasNext(); ) {
                x = i.next();
                if (!(a.contains(x) || b.contains(x))) {
                    throw new RuntimeException("Union: missing element");
                }
            }

            if (this.a != a || this.b != b) {
                throw new RuntimeException("Equality: proof/object mismatch");
            }
        }
    }

    public static class UnionBundle {
        public final Set a; 
        public final Set b;
        public final Set aub;
        public final IsUnion proof;

        public UnionBundle(final Set a, 
                           final Set b,
                           final Set aub,
                           final IsUnion proof) {
            this.a = a; 
            this.b = b; 
            this.aub = aub; 
            this.proof = proof;
        }
    }

    public static UnionBundle union(final Set a, final Set b) {
        final Set aub = new HashSet();
        aub.addAll(a); aub.addAll(b);
        return new UnionBundle(a, b, aub, new IsUnion(a, b, aub));
    }


    public static UnionBundle unionNil(final Set a) {
        return new UnionBundle(a, NIL, a, new IsUnion(a, NIL, a));
    }

    public static EqBundle unionEqualSets1(final Set a, final Set b, 
                                           final Set aub,
                                           IsUnion unionProof, // a u b
                                           Eq eqProof) // a = b
    {
        eqProof.runtimeCheck(a,b);
        unionProof.runtimeCheck(a,b,aub);
        return new EqBundle(a, aub, new Eq(a, aub));
    }

    public static EqBundle unionEqualSets2(final Set a, final Set b, 
                                           final Set aub,
                                           IsUnion unionProof, // a u b
                                           Eq eqProof) // a = b
    {
        eqProof.runtimeCheck(a,b);
        unionProof.runtimeCheck(a,b,aub);
        return new EqBundle(b, aub, new Eq(b, aub));
    }

}
    