package primes.trusted;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet; 

import static primes.trusted.SetManager.*;

public class FactorManager {
    public static final class HasFactors {
        private HasFactors(long n, Set factors) {
            runtimeCheck(n, factors);
        }

        private static void runtimeCheck(long n, Set factors) {
            if (factors == null) {
                throw new RuntimeException("Factor: empty factors set");
            }
            Long l;
            for (Iterator i = factors.iterator(); i.hasNext();) {
                l = (Long) i.next();
                if (n % l.longValue() != 0) {
                    throw new RuntimeException("Factor: " + l + 
                                               " invalid factor of " + n);
                }
            }
        }
    }

    public static class FactorBundle {
        public final long n;
        public final Set factors;
        public final UnionBundle union;
        public final HasFactors proof;
        public FactorBundle(final long n, 
                            final Set factors, 
                            final UnionBundle union,
                            final HasFactors proof) {
            this.n = n; 
            this.factors = factors; 
            this.union = union;
            this.proof = proof;
        }

        public String toString() {
            StringBuffer buf = new StringBuffer("n=");
            buf.append(n);
            buf.append(", factors=[");
            for (Iterator i = factors.iterator(); i.hasNext();) {
                buf.append(i.next());
                if (i.hasNext()) {
                    buf.append(',');
                }
            }
            return buf.append(']').toString();
        }
    }

    public static final FactorBundle NO_UNIT_FACTORS = 
        new FactorBundle(1, NIL, 
                         unionNil(NIL),
                         new HasFactors(1, NIL));

    public static FactorBundle factor(final long n) {
        if (1 == n) {
            return new FactorBundle(n, NIL, 
                                    unionNil(NIL),
                                    new HasFactors(n, NIL));
        } else {
            final Set factors = decompose(n);
            return new FactorBundle(n, factors, 
                                    unionNil(factors),
                                    new HasFactors(n, factors)); 
        }
    }

    public static FactorBundle multiply(final long n1, 
                                        final Set f1, 
                                        final HasFactors p1, 
                                        final long n2,
                                        final Set f2, 
                                        final HasFactors p2) {
        if (1 == n1) {
            return new FactorBundle(n2, f2, 
                                    union(NIL, f2),
                                    p2);
        } else if (1 == n2) {
            return new FactorBundle(n1, f1, 
                                    union(f1, NIL),
                                    p1);
        } else {
            final long n = n1 * n2;
            final UnionBundle union = union(f1, f2);
            final Set factors = union.aub;
            return new FactorBundle(n, factors, union,
                                    new HasFactors(n, factors));
        }
    }

    private static Set decompose(long n) throws IllegalArgumentException {
        if (1 < n) {
            Set factors = new HashSet();
            if (2 == n) { factors.add(new Long(2)); }
            for (long i = 2; i <= n / i; i++) {
                while (0 == n % i) {
                    factors.add(new Long(i));
                    n /= i;
                }
            }
            if (1 < n) { factors.add(new Long(n)); }
            return factors;
        } else {
            throw new IllegalArgumentException("Argument not decomposable");
        }
    }
}


